/*
 * be_generate.cc
 */

#include <be_description.hh>
#include <be_translate.hh>
#include <be_classes.hh>
#include <idl.hh>
#include <idl_extern.hh>
#include <stdio.h>

void generate_context_info(be_operation *op, ostream &out)
{
  UTL_StrList *context = op->context();
  UTL_StrlistActiveIterator *i = new UTL_StrlistActiveIterator (context);

  out << "Context ";
  while (!(i->is_done())) {
    out << i->item()->get_string() << " ";
    i->next();
  }
  out << "\n";
}
    
/*
 *  determine_specific_type
 */
void
determine_specific_type(AST_Type *type, AST_Type **specific, int *node_type)
{
  switch (type->node_type()) {
  case AST_Decl::NT_typedef:
    AST_Typedef *def;
    def = AST_Typedef::narrow_from_decl(type);
    if (def == NULL) {
      cout << "Narrowing typedef failed in determine_specific_type\n";
      BE_abort();
    }
    else 
      determine_specific_type(def->base_type(), specific, node_type);
    break;
  case AST_Decl::NT_pre_defined:
  case AST_Decl::NT_enum:
  case AST_Decl::NT_struct:
  case AST_Decl::NT_string:
  case AST_Decl::NT_array:
  case AST_Decl::NT_interface:
  case AST_Decl::NT_union:
    *specific  = type;
    *node_type = type->node_type();
  case AST_Decl::NT_sequence:
    *specific  = type;
    *node_type = type->node_type();
    break;
  default:
    cout << "In determine_specific_type: type (";
    cout << (int)type->node_type();
    cout << ") not expected.\n";
    BE_abort();
  }
}


/*
 * generate_typecode_2string
 */
void 
generate_typecode_2string(int code, char **out)
{
  char *tmp = (char* ) malloc(20);
  // output current values associated with typecodes. This way AST_Decl
  // class can have in the future different values for node_type, but
  // old description files will still be useful.
  switch(code) {
  case AST_Decl::NT_module:
    sprintf(tmp," NT_module ");
    break;
  case AST_Decl::NT_root:
    sprintf(tmp," NT_root ");
    break;
  case AST_Decl::NT_interface_fwd:
    sprintf(tmp," NT_interface_fwd ");
    break;
  case AST_Decl::NT_const:
    sprintf(tmp," NT_const ");
    break;
  case AST_Decl::NT_except:
    sprintf(tmp," NT_except ");
    break;
  case AST_Decl::NT_attr:
    sprintf(tmp," NT_attr ");
    break;
  case AST_Decl::NT_op:
    sprintf(tmp," NT_op ");
    break;
  case AST_Decl::NT_argument:
    sprintf(tmp," NT_argument ");
    break;
  case AST_Decl::NT_union_branch:
    sprintf(tmp," NT_union_branch ");
    break;
  case AST_Decl::NT_field:
    sprintf(tmp," NT_field ");
    break;
  case AST_Decl::NT_enum:
    sprintf(tmp," NT_enum ");
    break;
  case AST_Decl::NT_typedef:
    sprintf(tmp," NT_typedef ");
    break;
  case AST_Decl::NT_enum_val:
    sprintf(tmp," NT_enum_val ");
    break;
  case AST_Decl::NT_pre_defined:
    sprintf(tmp," NT_pre_defined ");
    break;
  case AST_Decl::NT_struct:
    sprintf(tmp," NT_struct ");
    break;
  case AST_Decl::NT_string:
    sprintf(tmp," NT_string ");
    break;
  case AST_Decl::NT_sequence:
    sprintf(tmp," NT_sequence ");
    break;
  case AST_Decl::NT_array:
    sprintf(tmp," NT_array ");
    break;
  case AST_Decl::NT_interface:
    sprintf(tmp," NT_interface ");
    break;
  case AST_Decl::NT_union:
    sprintf(tmp," NT_union ");
    break;
  default:
    cout << "Unexpected type code (" << code << ") in generate_typecode\n";
    BE_abort();
  }    
  *out = tmp;
}

/*
 * generate_typecode
 */
void 
generate_typecode(int code, ostream &o)
{
  char *tmp;
  generate_typecode_2string(code, &tmp);
  o << tmp;
  free(tmp);
}

/*
 * generate_extra_type_parameters
 * This routine generates more detailed information to be present ind
 * the description file being generated. For example, for strings the
 * MAXLEN info is output
 */
void generate_extra_type_parameters(int t, AST_Type *type, ostream &o)
{
  switch (t){
  case AST_Decl::NT_pre_defined:
  case AST_Decl::NT_enum:
  case AST_Decl::NT_struct:
  case AST_Decl::NT_interface:
    break;
  case AST_Decl::NT_string:
    be_string *bs;
    bs = be_string::narrow_from_decl(type);
    if (bs == NULL) {
      cout << "Error in narrowing string (generate_attrib_description)\n";
      BE_abort();
    }
    else {
      char digits[6];
      be_expr_val       expr(bs->max_size(), bs->max_size()->ev()->et);
      expr.value_only_2string_ctranslation(digits);
      o << " MAXLEN " << digits;
    }
    break;
  case AST_Decl::NT_array:
    be_array *barray;
    barray = be_array::narrow_from_decl(type);
    if (barray == NULL) {
      cout << "Error in narrowing from type do array\n";
      BE_abort();
    }
    else {
      char *typename_var = NULL, *basetypename_var = NULL;
      char digits[6];
      
      typename_var = translate_typename_for_pure_name(barray->base_type(), typename_var);
      // Notice that the typename_var built with the call above holds
      // something like "long" if the base_type is a predefined type
      // (since it extracts the CORBA from the beginning). I will adjust
      // it in basetypename_var and use it as the type for _buffer and as
      // the basetypename information in the list of complex types being
      // kept. 
      AST_Type *specific_type;
      int node_type;
      determine_specific_type(barray->base_type(), &specific_type, &node_type);
      if (node_type == AST_Decl::NT_pre_defined
	  || node_type == AST_Decl::NT_sequence
	  || node_type == AST_Decl::NT_string) {
	basetypename_var = (char *) malloc(strlen(typename_var) + 7);
	strcpy(basetypename_var, "CORBA_");
	strcat(basetypename_var, typename_var);
      }
      else {
	basetypename_var = strdup(typename_var);
      }
      
      o << " NDIMS ";
      for (int i = 0; i< barray->n_dims(); i++) {
	be_expr_val expr(barray->dims()[i], barray->dims()[i]->ev()->et);
      
	expr.value_only_2string_ctranslation(digits);
	o << digits << " ";
      }
      o << " BaseTypeName " << basetypename_var;
      determine_specific_type(barray->base_type(), &type, &t);
      o << " BaseTypeCode ";
      generate_typecode(t, o);
      free(typename_var);
      free(basetypename_var);
    }
    break;      
  case AST_Decl::NT_sequence:
    be_sequence *bseq;
    bseq = be_sequence::narrow_from_decl(type);
    if (bseq == NULL) {
      cout << "Error in narrowing from type do sequence\n";
      BE_abort();
    }
    else {
      char  *basetypename_var, *typename_var = NULL;
      char digits[6];
      be_expr_val expr(bseq->max_size(), bseq->max_size()->ev()->et);
      
      expr.value_only_2string_ctranslation(digits);
      typename_var = translate_typename_for_pure_name(bseq->base_type(), typename_var);
      // Notice that the typename_var built with the call above holds
      // something like "long" if the base_type is a predefined type
      // (since it extracts the CORBA from the beginning). I will adjust
      // it in basetypename_var and use it as the type for _buffer and as
      // the basetypename information in the list of complex types being
      // kept. 
      AST_Type *specific_type;
      int node_type;
      determine_specific_type(bseq->base_type(), &specific_type, &node_type);
      if (node_type == AST_Decl::NT_pre_defined
	  || node_type == AST_Decl::NT_sequence
	  || node_type == AST_Decl::NT_string) {
	basetypename_var = (char *) malloc(strlen(typename_var) + 7);
	strcpy(basetypename_var, "CORBA_");
	strcat(basetypename_var, typename_var);
	free(typename_var);
      }
      else {
	basetypename_var = typename_var;
      }
      
      o << " MAXLEN " << digits << " BaseTypeName " << basetypename_var;
      determine_specific_type(bseq->base_type(), &type, &t);
      o << " BaseTypeCode ";
      generate_typecode(t, o);
    }
    break;      
  default:
    cout << "Error in  generate_extra_type_parameters: \n"
	 << "type for attribute (" << t
	 << ") not expected.\n";
    BE_abort();
  }
}
/*
** generate_attrib_description
**/

void
generate_attrib_description(be_interface *itfc, ostream &o)
{
  be_utl_att_listActiveIterator          *i;
  int                                     t;
  AST_Type                               *type;
  
  o << "/* List of Attributes */\n";

  i = new be_utl_att_listActiveIterator(itfc->get_list_of_attribs());
  while ((!(i->is_done()))) {
    // checking integrity of the list
    if (i->item()->node_type() != AST_Decl::NT_attr) {
      cout << "Error in generate_attrib_description: element in "
	   << "list is not an attribute\n";
      BE_abort();
    }
	
    o << "Attribute " << i->item()->local_name()->get_string()
      << " TypeName ";
    if (i->item()->is_mirror()) {
      o << "MIRROR ";
    }
    translate_type (i->item()->field_type(), o);
    determine_specific_type(i->item()->field_type(), &type, &t);
    o << " TypeCode ";
    generate_typecode(t, o);
    // Some types need to be described by more parameters
    generate_extra_type_parameters(t, type, o);
    // if readonly, output this info. But in order to know it, we have
    // to narrow it  to AST_attribute from AST_Field
    /*
    be_attribute *at;
    at = be_attribute::narrow_from_decl(i->item());
    if (at == NULL ) {
      // narrow failed 
      cout << "Narrow in generate_attrib_description failed\n";
      BE_abort();
    }
    else { //ok 
    */
    if (i->item()->readonly())
      o << " Readonly ";
    /*}*/
    o << "\n";
    i->next();
  }

  delete i;  

}



/*
 * generate_operation_description
 */

void 
generate_operation_description (be_interface *itfc, ostream &o)
{
  be_utl_operation_listActiveIterator          *i;
  int                                           t;
  AST_Type                                     *type;


  o << "/* List of operations */\n";

  i = new be_utl_operation_listActiveIterator(
		     itfc->get_list_of_operations());
  while ((!(i->is_done()))) {
    o << "Operation " << i->item()->local_name()->get_string()
      << " TypeName ";
    if (i->item()->is_mirror()) {
      o << "MIRROR ";
    }
    translate_type (i->item()->return_type(), o);
    determine_specific_type(i->item()->return_type(), &type, &t);
    o << " TypeCode ";
    generate_typecode(t, o);
    // Some types need to be described by more parameters
    generate_extra_type_parameters(t, type, o);
    o << "\n";

    /* generate description for arguments */
    UTL_ScopeActiveIterator          *ai;
    AST_Decl                         *d;
    be_argument                      *ba;

    ai = new UTL_ScopeActiveIterator (i->item(), UTL_Scope::IK_decls);
    
    while (!(ai->is_done())) {
      d = ai->item();
      if (d->node_type() == AST_Decl::NT_argument) {
	ba = be_argument::narrow_from_decl(d);
	if (ba == NULL) {  // narrowing failed
	  cout << "Error while narrowing argument in "
	       << "generate_operation_descriptions\n";
	  BE_abort();
	}
	else {
	  o << "Argument " << d->local_name()->get_string();
	  // in, out, inout
	  switch (ba->direction()){
	  case AST_Argument::dir_IN: 
	    o << " IN ";
	    break;
	  case AST_Argument::dir_OUT: 
	    o << " OUT ";
	    break;
	  case AST_Argument::dir_INOUT: 
	    o << " INOUT ";
	    break;
	  default:
	    cout << "Error in generate_operation_description: "
		 << "invalid direction (in, out, inout) for argument\n";
	    BE_abort();
	  }
	  o << " TypeName ";
	  translate_type (ba->field_type(), o);
	  o << " TypeCode ";
	  determine_specific_type(ba->field_type(), &type, &t);
	  generate_typecode(t, o);
	  // Some types need to be described by more parameters
	  generate_extra_type_parameters(t, type, o);
	  o << "\n";
	}
	ai->next();
      }
      else { // error
	cout <<"Error in generate_operation_description: expected argument!\n";
	BE_abort();
      }
    }
    delete ai;

    // information about context
    if (i->item()->context() != NULL)
      generate_context_info(i->item(), o);  
    i->next();
  }
  
  delete i;  
}


