/* 
 * be_classes.cc
 *
 * This is part of Georgia Tech COBS project's IDL to C compiler
 * implementation. This implementation was built on top of the compiler
 * front end freely available from SunSoft, therefore we include here
 * their copyright notice.
 *
 */


/*

COPYRIGHT

Copyright 1992, 1993, 1994 Sun Microsystems, Inc.  Printed in the United
States of America.  All Rights Reserved.

This product is protected by copyright and distributed under the following
license restricting its use.

The Interface Definition Language Compiler Front End (CFE) is made
available for your use provided that you include this license and copyright
notice on all media and documentation and the software program in which
this product is incorporated in whole or part. You may copy and extend
functionality (but may not remove functionality) of the Interface
Definition Language CFE without charge, but you are not authorized to
license or distribute it to anyone else except as part of a product or
program developed by you or with the express written consent of Sun
Microsystems, Inc. ("Sun").

The names of Sun Microsystems, Inc. and any of its subsidiaries or
affiliates may not be used in advertising or publicity pertaining to
distribution of Interface Definition Language CFE as permitted herein.

This license is effective until terminated by Sun for failure to comply
with this license.  Upon termination, you shall destroy or return all code
and documentation for the Interface Definition Language CFE.

INTERFACE DEFINITION LANGUAGE CFE IS PROVIDED AS IS WITH NO WARRANTIES OF
ANY KIND INCLUDING THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS
FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR ARISING FROM A COURSE OF
DEALING, USAGE OR TRADE PRACTICE.

INTERFACE DEFINITION LANGUAGE CFE IS PROVIDED WITH NO SUPPORT AND WITHOUT
ANY OBLIGATION ON THE PART OF Sun OR ANY OF ITS SUBSIDIARIES OR AFFILIATES
TO ASSIST IN ITS USE, CORRECTION, MODIFICATION OR ENHANCEMENT.

SUN OR ANY OF ITS SUBSIDIARIES OR AFFILIATES SHALL HAVE NO LIABILITY WITH
RESPECT TO THE INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY
INTERFACE DEFINITION LANGUAGE CFE OR ANY PART THEREOF.

IN NO EVENT WILL SUN OR ANY OF ITS SUBSIDIARIES OR AFFILIATES BE LIABLE FOR
ANY LOST REVENUE OR PROFITS OR OTHER SPECIAL, INDIRECT AND CONSEQUENTIAL
DAMAGES, EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

Use, duplication, or disclosure by the government is subject to
restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in
Technical Data and Computer Software clause at DFARS 252.227-7013 and FAR
52.227-19.

Sun, Sun Microsystems and the Sun logo are trademarks or registered
trademarks of Sun Microsystems, Inc.

SunSoft, Inc.  
2550 Garcia Avenue 
Mountain View, California  94043

NOTE:

SunOS, SunSoft, Sun, Solaris, Sun Microsystems or the Sun logo are
trademarks or registered trademarks of Sun Microsystems, Inc.

 */

#pragma ident "%@(#)be_classes.cc	1.20% %92/06/10% Sun Microsystems"

// be_classes.cc
//
// Implementation of all dummy BE classes

#include        <malloc.h>
#include        <string.h>
#include        <stdio.h>
#include	<idl.hh>
#include	<idl_extern.hh>
#include	<be.hh>
#include        <be_translate.hh>
#include        <be_description.hh>
#include        "be_type_description.hh"
#include        "assert.h"

void BE_go_through_declarations(  UTL_ScopeActiveIterator    *i, 
				  ostream &fout, ostream &descout);

/*
 * BE_PredefinedType
 */
be_predefined_type::be_predefined_type()
{
}
be_predefined_type::be_predefined_type(AST_PredefinedType::PredefinedType t,
				       UTL_ScopedName *n, UTL_StrList *p)
		  : AST_PredefinedType(t, n, p),
		    AST_Decl(AST_Decl::NT_pre_defined, n, p)
{
}

/*
 * Ctranslation for BE_predefined_type
 */
void
be_predefined_type::ctranslation(ostream &o)
{
  char *out;
  out = ctranslation2dstring();
  o << out << " ";
  free(out);
}

/*
 * Ctranslation for BE_predefined_type stored in a string
 */
char *
be_predefined_type::ctranslation2dstring()
{
  char *o;
  switch (pt()) {
  case PT_long:
    o = strdup("CORBA_long");
    break;
  case PT_ulong:
    o = strdup("CORBA_unsigned_long");
    break;
  case PT_longlong:
    // so far I am mapping long long into long
    o = strdup("CORBA_long");
    break;
  case PT_ulonglong:
    // so far I am mapping long long into long
    o = strdup("CORBA_long");
    break;
  case PT_short:
    o = strdup("CORBA_short");
    break;
  case PT_ushort:
    o = strdup("CORBA_unsigned_short");
    break;
  case PT_float:
    o = strdup("CORBA_float");
    break;
  case PT_double:
    o = strdup("CORBA_double");
    break;
  case PT_longdouble:
    // so far I am mapping longdouble into double
    o = strdup("CORBA_double");
    break;
  case PT_char:
    o = strdup("CORBA_char");
    break;
  case PT_wchar:
    // so far I am mapping wchar into char
    o = strdup("CORBA_char");
    break;
  case PT_boolean:
    o = strdup("CORBA_boolean");
    break;
  case PT_octet:
    o = strdup("CORBA_octet");
    break;
  case PT_any:
    o = strdup("CORBA_any");
    break;
  case PT_void:
    o = strdup("CORBA_void");
    break;
  case PT_pseudo:
    // in fe_init.cc PT_pseudo are created for lots of things (actually all
    // reserved words ... Here I may want to get only "Object", but so far
    // I will print out whatever it holds, with "CORBA_ in front of it
    o = (char*) malloc(strlen("CORBA_") + strlen(local_name()->get_string()) + 1);
    strcpy(o, "CORBA_");
    strcat(o, local_name()->get_string());
    break;
  default:
cout <<"This shouldn't be reached in "
     <<"be_predefined_type::ctranslation2dstring\npt() is "
     <<(int)pt() << "\n";
    BE_abort();
  }
  return o;
}

void
be_predefined_type::build_type_info()
{
}

// Narrowing
IMPL_NARROW_METHODS1(be_predefined_type, AST_PredefinedType)
IMPL_NARROW_FROM_DECL(be_predefined_type)



/*
 * BE_Module
 */
be_module::be_module()
{
}
be_module::be_module(UTL_ScopedName *n, UTL_StrList *p)
	 : AST_Decl(AST_Decl::NT_module, n, p),
	   UTL_Scope(AST_Decl::NT_module)
{
}

void
be_module::build_type_info()
{
}

void
be_module::ctranslation2(ostream &o, ostream &descout) 
{
  UTL_ScopeActiveIterator    *i;
  // Modules are used only to specify scope, so the only thing we have
  // to do in order to generate code/specs for a Module is to go through
  // its list of declarations.
  o << "/* Module ";
  translate_scoped_name(this, o);
  o << "  */\n";
  i = new UTL_ScopeActiveIterator(this, UTL_Scope::IK_decls);
  BE_go_through_declarations(i, o, descout);
  delete i;
  o << "/* end of Module ";
  translate_scoped_name(this, o);
  o << "  */\n";
}

// Narrowing
IMPL_NARROW_METHODS1(be_module, AST_Module)
IMPL_NARROW_FROM_DECL(be_module)
IMPL_NARROW_FROM_SCOPE(be_module)

/*
 * BE_Interface
 */
be_interface::be_interface() 
             : pd_already_defined(I_FALSE),
	       pd_def_list(NULL),
	       pd_attrib_list(NULL), pd_operation_list(NULL)
{
}
be_interface::be_interface(UTL_ScopedName *n, AST_Interface **ih, long nih,
			   UTL_StrList *p)
	    : pd_already_defined(I_FALSE),
	      pd_def_list(NULL),
	      pd_attrib_list(NULL),
	      AST_Interface(n, ih, nih, p),
	      pd_operation_list(NULL),
	      AST_Decl(AST_Decl::NT_interface, n, p),
	      UTL_Scope(AST_Decl::NT_interface)
{
}

/* Operations related to keeping a list of identifieres already
 * defined for this interface
 */
void
be_interface::add_to_list_of_definitions(Identifier *name)
{
  pd_def_list = new UTL_IdList(name, pd_def_list);
}

idl_bool
be_interface::is_already_defined(Identifier *name)
{
  UTL_IdListActiveIterator          *i;
  idl_bool                          found = I_FALSE;

  i = new UTL_IdListActiveIterator(pd_def_list);

  while ((!(i->is_done())) && (!found) ) {
    if (i->item()->compare(name)== I_TRUE)
      found = I_TRUE;
    else
      i->next();
  }

  delete i;
  return found;

}


/* The two following operations are related to keeping a list of
 * attributes associated to the interface (for subsequent generation
 * of C code for _set and _get)
 */


void
be_interface::add_to_list_of_attribs (be_attribute *att) 
{
  pd_attrib_list = new be_utl_att_list(att, pd_attrib_list);
}

be_utl_att_list *
be_interface::get_list_of_attribs()
{
  return pd_attrib_list;
}


// The following two functions keep list of operations for the
// interface. See explanation  of why we want this in be_classes.hh

void
be_interface::add_to_list_of_operations (be_operation *op) 
{
  pd_operation_list = new be_utl_operation_list(op, pd_operation_list);
}

be_utl_operation_list *
be_interface::get_list_of_operations()
{
  return pd_operation_list;
}



extern void 
generate_typecode(int code, ostream &o);

/*
 * Ctranslation
 */

void
be_interface::ctranslation(ostream &o)
{
  if (pd_already_defined || imported()) {
    translate_scoped_name (this, o);
  }
  else {  
    // it may be the case where this interface was already declared
    // through an interface_forward. Therefore I will declare this
    // "conditionally".
    o << "\n#ifndef _CORBA_interface_";
    translate_scoped_name(this,o);
    o << "\n#define _CORBA_interface_";
    translate_scoped_name(this,o);
    o << "\n";
    o << "typedef CORBA_Object ";
    translate_scoped_name(this, o);
    o << ";\n#endif\n";

    // now the declarations inside this have to be output ...
    o << "\n/****** Declarations for interface ";
    translate_scoped_name(this,o);
    o << "******/\n";

    UTL_ScopeActiveIterator             *i;
    AST_Decl                            *d;
    i = new UTL_ScopeActiveIterator (this, UTL_Scope::IK_decls);
    while (!(i->is_done())) {
      d = i->item();

      d->ctranslation(o);

      // keep the information that this identifier was declared
      add_to_list_of_definitions(d->local_name());

      // deals with the list of attributes being kept by interface
      if (d->node_type() == AST_Decl::NT_attr) { 
	// it just translated an attribute
	be_attribute *att;
	att = be_attribute::narrow_from_decl(d);
	if (att == NULL) { // error while narrowing
	  cout << "Error while narrowing attribute in "
	       << "be_interface:ctranslation\n";
	  BE_abort();
	}
	else // narrowing went ok
	  add_to_list_of_attribs (att);
      }

      // deals with the list of operations being kept by interface
      if (d->node_type() == AST_Decl::NT_op) { 
	// it just translated an operation
	be_operation *op;
	op = be_operation::narrow_from_decl(d);
	if (op == NULL) { // error while narrowing
	  cout << "Error while narrowing operation in "
	       << "be_interface:ctranslation\n";
	  BE_abort();
	}
	else // narrowing went ok
	  add_to_list_of_operations (op);
      }

	
      i->next();
    }
    delete i;  
  }

  if (!pd_already_defined) {
      build_type_info();
      pd_already_defined = I_TRUE;
  }
  /* After iterating through this interface's declarations, let's
     define operations/types/const/etc inherited */
  if (n_inherits() > 0)
    translate_for_inherited_interfaces(this, inherits(),
				       n_inherits(), o);

  o << "extern ";
  translate_scoped_name(this,o);
  o << "  COBS_";
  translate_scoped_name(this,o);
  o << "_object_create();\n";
  o << "extern ";
  translate_scoped_name(this,o);
  o << "  ";
  translate_scoped_name(this,o);
  o << "__Create(void *instanceData);\n";
}

void
be_interface::build_type_info()
{
    char *tname;
    char *tinfo;

    tname = translate_scoped_name2dstring(this);
    tinfo = (char*) malloc(strlen("TYPEDEF_OF CORBA_Object") + 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF CORBA_Object");
    be_description_type_list->add(tname, tinfo);
}

// Narrowing
IMPL_NARROW_METHODS1(be_interface, AST_Interface)
IMPL_NARROW_FROM_DECL(be_interface)
IMPL_NARROW_FROM_SCOPE(be_interface)

/*
 * BE_InterfaceFwd
 */
be_interface_fwd::be_interface_fwd() : pd_already_defined(I_FALSE)
{
}
be_interface_fwd::be_interface_fwd(UTL_ScopedName *n, UTL_StrList *p)
		: pd_already_defined(I_FALSE),
		  AST_InterfaceFwd(n, p),
		  AST_Decl(AST_Decl::NT_interface_fwd, n, p)
{
}

/*
 * Ctranslation
 */

void
be_interface_fwd::ctranslation(ostream &o)
{
  if (!pd_already_defined || imported()) {
    o << "\n/****** Interface Forward ******/\n";
    // the #define bellow exists in order to avoid that the actual
    // definition related to this declaration end up generating this
    // typedef again. Notice that using an interface forward when
    // the interface was already defined will generate a translation in C
    // with two typedef, and therefore erroneous.
    o << "#define _CORBA_interface_";
    translate_scoped_name(this, o);    
    o << "\ntypedef CORBA_Object ";
    translate_scoped_name(this, o);
    o << ";\n";
  }
  if (!pd_already_defined) {
      build_type_info();
      pd_already_defined = I_TRUE;
  }
}
  
void
be_interface_fwd::build_type_info()
{
    char *tname;
    char *tinfo;

    tname = translate_scoped_name2dstring(this);
    tinfo = (char*) malloc(strlen("TYPEDEF_OF CORBA_Object") + 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF CORBA_Object");
    be_description_type_list->add(tname, tinfo);
}

// Narrowing
IMPL_NARROW_METHODS1(be_interface_fwd, AST_InterfaceFwd)
IMPL_NARROW_FROM_DECL(be_interface_fwd)

/*
 * BE_Exception
 */
be_exception::be_exception()
{
}
be_exception::be_exception(UTL_ScopedName *n, UTL_StrList *p)
	    : AST_Decl(AST_Decl::NT_except, n, p),
              AST_Structure(AST_Decl::NT_except, n, p),
	      UTL_Scope(AST_Decl::NT_except)
{
}

/*
 * Ctranslation for exceptions
 */
long be_exception::id = 0;

void
be_exception::ctranslation(ostream &o)
{
  int nb_fields;
  char **array_field_description;

  translate_field_types((UTL_Scope *) this, o);
  o << "#define ex_";
  translate_scoped_name ((AST_Decl *) this, o);
  o << "   \"" << id++ << "\"\n";  
  o << "typedef struct ";
  translate_scoped_name ((AST_Decl *) this, o);
  o << " {\n";
  translate_fields((UTL_Scope *) this, o, &nb_fields, 
		   &array_field_description, 1);
  o << "  int not_used; /* in case no fields above */\n";
  o << "} ";
  translate_scoped_name ((AST_Decl *) this, o);
  o << ";\n";
}


void
be_exception::build_type_info()
{
    int nb_fields;
    char **array_field_description;

    translate_fields ((UTL_Scope *) this, cerr, &nb_fields, &array_field_description, 0);
}

// Narrowing
IMPL_NARROW_METHODS1(be_exception, AST_Exception)
IMPL_NARROW_FROM_DECL(be_exception)
IMPL_NARROW_FROM_SCOPE(be_exception)

/*
 * BE_Structure
 */
be_structure::be_structure() : pd_already_defined(I_FALSE)
{
}
be_structure::be_structure(UTL_ScopedName *n, UTL_StrList *p)
             :  pd_already_defined(I_FALSE),
		AST_Decl(AST_Decl::NT_struct, n, p),
		UTL_Scope(AST_Decl::NT_struct)
{
}

/* make_not_defined and make_defined are used in
 * translate_from_inherited_interfaces
 */ 
void
be_structure::make_not_defined()
{
  pd_already_defined = I_FALSE;
}

void
be_structure::make_defined()
{
  pd_already_defined = I_TRUE;
}


idl_bool
be_structure::already_defined()
{
  return pd_already_defined;
}

/*
 * Ctranslation for be_structure
 */
void
be_structure::ctranslation(ostream &o)
{
  if (pd_already_defined || imported()) { // don't need to define the structure again
    translate_scoped_name ((AST_Decl *) this, o);
    o << " ";
  }
  else {
    int nb_fields;
    char **array_field_description;

    translate_field_types((UTL_Scope *) this, o);

    o << "typedef struct ";
    o << " {\n";
    translate_fields ((UTL_Scope *) this, o, &nb_fields, &array_field_description, 1);
    idl_global->indent()->skip_to(o);
    o << "} ";
    translate_scoped_name ((AST_Decl *) this, o);
    o << ";\n";
  }

  if (!pd_already_defined) {
      build_type_info();
      pd_already_defined = I_TRUE;
  }
}

void
be_structure::build_type_info()
{
    // adding information about this type to list of complex types
    int nb_fields;
    char **array_field_description;

    translate_fields ((UTL_Scope *) this, cerr, &nb_fields, &array_field_description, 0);
    char *tname;
    tname = translate_scoped_name2dstring ((AST_Decl *) this);

    int i;
    int tinfo_size = strlen("NT_struct NB_FIELDS ") + 6; // assuming 6 is
                                                          // more than enough
                                                          // for digits
    for (i=0; i < nb_fields; i++)
      tinfo_size += strlen(array_field_description[i]) + 1;
    char *tinfo = (char *) malloc(tinfo_size+1);
    assert(tinfo != NULL);
    sprintf(tinfo,"NT_struct NB_FIELDS %d ", nb_fields);
    for (i=0; i < nb_fields; i++) {
      strcat(tinfo, array_field_description[i]);
      strcat(tinfo, " ");
      free(array_field_description[i]);
    }
    free(array_field_description);
    be_description_type_list->add(tname, tinfo);
}

// Narrowing
IMPL_NARROW_METHODS1(be_structure, AST_Structure)
IMPL_NARROW_FROM_DECL(be_structure)
IMPL_NARROW_FROM_SCOPE(be_structure)

/*
 * BE_Enum
 */
be_enum::be_enum() : pd_already_defined(I_FALSE)
{
}
be_enum::be_enum(UTL_ScopedName *n, UTL_StrList *p)
       : pd_already_defined(I_FALSE),
	 AST_Enum(n, p),
	 AST_Decl(AST_Decl::NT_enum, n, p),
	 UTL_Scope(AST_Decl::NT_enum)
{
}

/* make_not_defined and make_defined are used in
 * translate_from_inherited_interfaces
 */ 

void
be_enum::make_not_defined()
{
  pd_already_defined = I_FALSE;
}

void
be_enum::make_defined()
{
  pd_already_defined = I_TRUE;
}


/*
 * Ctranslation for BE_enum
 */

void
be_enum::ctranslation(ostream &o)
{
  if (pd_already_defined || imported()) {
    // already translated in this file
    translate_scoped_name((AST_Decl *) this, o);
    o << " ";
  }
  else {
    o << "typedef CORBA_enum ";
    translate_scoped_name((AST_Decl *) this, o);
    o << ";\n";
  }
  if (!pd_already_defined) {
      // output information about type enum
      build_type_info();
      pd_already_defined = I_TRUE;
  }
}

void
be_enum::build_type_info()
{
    char *tname = translate_scoped_name2dstring(this);
    char *tinfo; 
    tinfo = (char*) malloc(strlen("NT_enum") + 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "NT_enum");
    be_description_type_list->add(tname, tinfo);
}

// Narrowing
IMPL_NARROW_METHODS1(be_enum, AST_Enum)
IMPL_NARROW_FROM_DECL(be_enum)
IMPL_NARROW_FROM_SCOPE(be_enum)

/*
 * BE_Operation
 */
be_operation::be_operation()
{
}
be_operation::be_operation(AST_Type *rt, AST_Operation::Flags fl,
			   UTL_ScopedName *n, UTL_StrList *p)
	    : AST_Operation(rt, fl, n, p),
	      AST_Decl(AST_Decl::NT_op, n, p),
	      UTL_Scope(AST_Decl::NT_op)
{
}

void
be_operation::ctranslation (ostream &o)
{
  char st[255];
  translate_argument_types(this, o);

  if (return_type()->imported()) {
    return_type()->build_type_info();
  }
  for (int i = 0; i< 2; i++) {
    o << "extern ";
    translate_type(return_type(), o);
    o << " ";
    AST_Type *junk;
    int node_type;
    determine_specific_type(return_type(), &junk, &node_type);
    switch(node_type) {
    case AST_Decl::NT_array:
    case AST_Decl::NT_sequence:
      o << "*";
      break;
    default:
      ;
    }
    if (i== 1) o << "impl_";
    translate_scoped_name(this, o);
    o << "(";
    translate_scope_part(this, st);
    o << st << " object";
    if ( context() != NULL)
      o << ", CORBA_Context ct";
    /* arguments */
    translate_arguments(this, o);
    o << ", CORBA_Environment *ev);\n";
  }
}

// Narrowing
IMPL_NARROW_METHODS1(be_operation, AST_Operation)
IMPL_NARROW_FROM_DECL(be_operation)
IMPL_NARROW_FROM_SCOPE(be_operation)

/*
 * BE_Field
 */
be_field::be_field()
{
}
be_field::be_field(AST_Type *ft, UTL_ScopedName *n, UTL_StrList *p)
	: AST_Field(ft, n, p),
	  AST_Decl(AST_Decl::NT_field, n, p)
{
}

// Narrowing
IMPL_NARROW_METHODS1(be_field, AST_Field)
IMPL_NARROW_FROM_DECL(be_field)

/*
 * BE_Argument
 */
be_argument::be_argument()
{
}
be_argument::be_argument(AST_Argument::Direction d, AST_Type *ft,
			 UTL_ScopedName *n, UTL_StrList *p)
	   : AST_Argument(d, ft, n, p),
	     AST_Field(AST_Decl::NT_argument, ft, n, p),
	     AST_Decl(AST_Decl::NT_argument, n, p)
{
}

void
be_argument::ctranslation(ostream &o)
{
  AST_Type *type = field_type();
  AST_Type *specific_type;
  int      node_type;

  // If it is a typedef or something like that, needs to nail town to
  // basic type so we can know what to do.
  determine_specific_type(type, &specific_type, &node_type);

  switch (direction()) {
  case AST_Argument::dir_IN: {
    switch (node_type) {
    case AST_Decl::NT_interface:
    case AST_Decl::NT_string:
    case AST_Decl::NT_pre_defined:
    case AST_Decl::NT_enum:      
      //      translate_type(specific_type, o);
      translate_type(type, o);
      break;
    case AST_Decl::NT_struct:
    case AST_Decl::NT_union:
    case AST_Decl::NT_sequence:
      //      translate_type(specific_type, o);
      translate_type(type, o);
      o << " *";
      break;
    case AST_Decl::NT_array: {
      AST_Array *array;
      array = AST_Array::narrow_from_decl(specific_type);
      if (array == NULL) { // narrow failed
	cout << "Error while translating array argument\n";
	BE_abort();
      }
      else {
	translate_type(type, o);
	o << " *";
      }
      break;
    }
    default:
      cout << "In be_argument::ctranslation: not expected type of argument ("
	   << node_type << ")\n";
      BE_abort();
      break;
    } // end of switch in terms of node_type
    break;
  } // end of dir_IN

  case AST_Argument::dir_OUT:
  case AST_Argument::dir_INOUT:{
    switch (node_type) {
    case AST_Decl::NT_interface:
    case AST_Decl::NT_string:
    case AST_Decl::NT_pre_defined:
    case AST_Decl::NT_enum:      
    case AST_Decl::NT_struct:
    case AST_Decl::NT_union:
    case AST_Decl::NT_sequence:
      //      translate_type(specific_type, o);
      translate_type(type, o);
      o << " *";
      break;
    case AST_Decl::NT_array:
      AST_Array *array;
      array = AST_Array::narrow_from_decl(specific_type);
      if (array == NULL) { // narrow failed
	cout << "Error while translating array argument\n";
	BE_abort();
      }
      else {
	translate_type(type, o);
	o << " *";
      }
      break;
    default:
      cout << "In be_argument::ctranslation: not expected type of argument ("
	   << node_type << ")\n";
      BE_abort();
      break;
    } // end of switch in terms of node_type
    break;
  }
  default:
    cout << "Error in be_argument: invalid direction (in, out, inout)\n";
    BE_abort();
  }
  // name
  o << local_name()->get_string();
} // end of be_argument::ctranslation


// Narrowing
IMPL_NARROW_METHODS1(be_argument, AST_Argument)
IMPL_NARROW_FROM_DECL(be_argument)

/*
 * BE_Attribute
 */
be_attribute::be_attribute()
{
}
be_attribute::be_attribute(idl_bool ro, AST_Type *ft, UTL_ScopedName *n,
			   UTL_StrList *p)
	    : AST_Attribute(ro, ft, n, p),
	      AST_Field(AST_Decl::NT_attr, ft, n, p),
	      AST_Decl(AST_Decl::NT_attr, n, p)
{
}

/*
 * Ctranslation for be_attribute
 */
void
be_attribute::ctranslation(ostream &o)
{
  if (field_type()->imported()) {
    field_type()->build_type_info();
  }
  predefine_type(field_type(), o);
  o << "extern ";
  translate_type(field_type(), o);
  o << " ";

  // Since attributes are part of interfaces which can't be
  // nested, the scope part of the attribute is just the
  // interface local name. But at this point we don't directly
  // have information about which interface we are translating for,
  // so I use translate_scope_part in order to get it.
  be_idlist namebe (name());
  char attscope[255];
  AST_Type *junk;
  int node_type;
  determine_specific_type(field_type(), &junk, &node_type);
  switch(node_type) {
  case AST_Decl::NT_array:
  case AST_Decl::NT_sequence:
    o << "*";
    break;
  default:
    ;
  }
  namebe.translate_scope_part(attscope);
  o << attscope << "__get_" << local_name()->get_string() << "("
    << attscope << " o, CORBA_Environment *ev);\n";
  if (!readonly()) {
    o << "extern void ";
    o << attscope << "__set_" << local_name()->get_string() 
      << "(" << attscope << " o, ";
    translate_type (field_type (), o);
    switch (node_type) {
    case AST_Decl::NT_struct:
    case AST_Decl::NT_sequence:
    case AST_Decl::NT_array:
      o << " *v, CORBA_Environment *ev);\n";
      break;
    default:
      o <<" v, CORBA_Environment *ev);\n";
      break;
    }
  }

}

// Narrowing
IMPL_NARROW_METHODS1(be_attribute, AST_Attribute)
IMPL_NARROW_FROM_DECL(be_attribute)

/*
 * BE_Union
 */
be_union::be_union() : pd_already_defined(I_FALSE)
{
}
be_union::be_union(AST_ConcreteType *dt, UTL_ScopedName *n, UTL_StrList *p)
	: pd_already_defined(I_FALSE),
	  AST_Union(dt, n, p),
	  AST_Decl(AST_Decl::NT_union, n, p),
          AST_Structure(AST_Decl::NT_union, n, p),
	  UTL_Scope(AST_Decl::NT_union)
{
}

/* make_not_defined and make_defined are used in
 * translate_from_inherited_interfaces
 */ 

void
be_union::make_not_defined()
{
  pd_already_defined = I_FALSE;
}

void
be_union::make_defined()
{
  pd_already_defined = I_TRUE;
}

/*
 * Ctranslation for union 
 */
void
be_union::ctranslation(ostream &o)
{
  if (pd_already_defined || imported()) {
    translate_scoped_name(this,o);
    o << " ";
  }
  else {
    o << "typedef struct {\n";
    if (idl_global->indent() == NULL)
      idl_global->set_indent(new UTL_Indenter());
    idl_global->indent()->increase();
    idl_global->indent()->skip_to(o);
    translate_type(disc_type(),o);
    o << " _d;\n";
    idl_global->indent()->skip_to(o);
    o  << "union {\n";
    translate_union_branches(this, o);
    idl_global->indent()->skip_to(o);
    o << "} _u;\n";
    idl_global->indent()->decrease();
    idl_global->indent()->skip_to(o);
    o << "} ";
    translate_scoped_name(this, o);
    o << ";\n";
    pd_already_defined = I_TRUE;
  }
}

// Narrowing
IMPL_NARROW_METHODS1(be_union, AST_Union)
IMPL_NARROW_FROM_DECL(be_union)
IMPL_NARROW_FROM_SCOPE(be_union)

/*
 * BE_UnionBranch
 */
be_union_branch::be_union_branch()
{
}
be_union_branch::be_union_branch(AST_UnionLabel *lab, AST_Type *ft,
				 UTL_ScopedName *n, UTL_StrList *p)
	       : AST_UnionBranch(lab, ft, n, p),
		 AST_Field(AST_Decl::NT_union_branch, ft, n, p),
		 AST_Decl(AST_Decl::NT_union_branch, n, p)
{
}

/* 
 * Ctranslation
 */
void
be_union_branch::ctranslation(ostream &o)
{
  translate_type(field_type(), o);
  o << local_name()->get_string();
}

// Narrowing
IMPL_NARROW_METHODS1(be_union_branch, AST_UnionBranch)
IMPL_NARROW_FROM_DECL(be_union_branch)

/*
 * BE_UnionLabel
 */
be_union_label::be_union_label()
{
}
be_union_label::be_union_label(AST_UnionLabel::UnionLabel ul,
			       AST_Expression *v)
	      : AST_UnionLabel(ul, v)
{
}

/*
 * BE_Constant
 */
be_constant::be_constant()
{
}
be_constant::be_constant(AST_Expression::ExprType et,
			 AST_Expression *v,
			 UTL_ScopedName *n,
			 UTL_StrList *p)
	   : AST_Constant(et, v, n, p),
	     AST_Decl(AST_Decl::NT_const, n, p)
{
}

void 
be_constant::ctranslation(ostream &o) {
	//  UTL_ScopedName                 *name;
  be_expr_val                    expr(constant_value(), et());

  // o << "/* Translating from line " << line()  << " */\n";
  o << "#define ";
  
  translate_scoped_name( (AST_Decl *) this, o);
  o << "   ";

  expr.ctranslation(o);

  o << "\n";
}

// Narrowing
IMPL_NARROW_METHODS1(be_constant, AST_Constant)
IMPL_NARROW_FROM_DECL(be_constant)

/*
 * BE_Expression
 */
be_expr_val::be_expr_val(UTL_ScopedName *n) : AST_Expression(n) {}
be_expr_val::be_expr_val(AST_Expression *b, AST_Expression::ExprType t)
	   : AST_Expression(b, t) 
{
}
be_expr_val::be_expr_val(AST_Expression::ExprComb c,
			 AST_Expression *v1,
			 AST_Expression *v2)
	   : AST_Expression(c, v1, v2)
{
}
be_expr_val::be_expr_val(long l) : AST_Expression(l) {}
be_expr_val::be_expr_val(long l, AST_Expression::ExprType t)
	   : AST_Expression(l, t)
{
}
be_expr_val::be_expr_val(unsigned long l) : AST_Expression(l) {}
be_expr_val::be_expr_val(String *s) : AST_Expression(s) {}
be_expr_val::be_expr_val(char c) : AST_Expression(c) {}
be_expr_val::be_expr_val(double d) : AST_Expression(d) {}

void
be_expr_val::ctranslation(ostream &o) {
  AST_Expression::AST_ExprValue  *exprvalue;

  exprvalue = this->eval(AST_Expression::EK_const);
  switch (exprvalue->et) {
  case AST_Expression::EV_short: 	    // Expression value is short
    o << "( (CORBA_short) " << exprvalue->u.sval << " )";
    break;
  case AST_Expression::EV_ushort:	// Expression value is unsigned short
    o << "( (CORBA_unsigned_short) " << exprvalue->u.usval << " )";
    break;
  case AST_Expression::EV_long:			// Expression value is long
    o << "( (CORBA_long) " << exprvalue->u.lval << " )";
    break;
  case AST_Expression::EV_ulong:	// Expression value is unsigned long
    o << "( (CORBA_unsigned_long) " << exprvalue->u.ulval << " )";
    break;
  case AST_Expression::EV_longlong:	// Expression value is long long
    // I am going to map this to long ... there is no reference to a 
    // CORBA_long_long ...
    o << "( (CORBA_long) " << exprvalue->u.lval << " )";
    break;
  case AST_Expression::EV_ulonglong:  // Expression value is unsigned long long
    // Dito as above
    o << "( (CORBA_unsigned_long) " << exprvalue->u.ulval << " )";
    break;
  case AST_Expression::EV_float:	// Expression value is 32-bit float
    o << "( (CORBA_float) " << exprvalue->u.fval << " )";
    break;
  case AST_Expression::EV_double:	// Expression value is 64-bit float
    o << "( (CORBA_double) " << exprvalue->u.dval << " )";
    break;
  case AST_Expression::EV_longdouble:	// Expression value is 128-bit float
    // nothing about this in the CORBA document I have. I will just map to
    // a 64-bit float, it is going to work for our pratical proposals.
    o << "( (CORBA_double) " << exprvalue->u.dval << " )";
    break;
  case AST_Expression::EV_char:		// Expression value is char
    o << "( (CORBA_char) " << exprvalue->u.cval << " )";
    break;
  case AST_Expression::EV_wchar:	// Expression value is wide char   
    // again, I don't know anything about this.
    o << "( (CORBA_char) " << exprvalue->u.cval << " )";
    break;
  case AST_Expression::EV_octet:	// Expression value is unsigned char
    o << "( (CORBA_octet) " << exprvalue->u.oval << " )";
    break;
  case AST_Expression::EV_bool:		// Expression value is boolean
    o << "( (CORBA_boolean) " << exprvalue->u.bval << " )";
    break;
  case AST_Expression::EV_string:	// Expression value is char *
    o << "( (*CORBA_char) " << exprvalue->u.strval << " )";
    break;
  case AST_Expression::EV_wstring:	// Expression value is wchar_t *
    o << "( (*CORBA_char) " << exprvalue->u.strval << " )";
    break;
  case AST_Expression::EV_any:		// Expression value is any of above
    // I don't see the meaning for constant, so I won't generate anything.
    // I started trying to generate (CORBA_any) blablabla, but there is
    // no value, there is no field in the struct exprvalue to get it from.
    break;
  case AST_Expression::EV_void:		// Expression value is void (absent)
    break;
  case AST_Expression::EV_none:		// Expression value is missing
    break;
  default: {
    // this is not expected to happen, since it would be an IDL error
    cout << "Error in  be_expr_val::ctranslation\n";
    BE_abort();
    }
  }
}

void
be_expr_val::value_only_ctranslation(ostream &o) 
{
  char digits[6];
  value_only_2string_ctranslation(digits);
  o << digits;
}

void
be_expr_val::value_only_2string_ctranslation(char *o) {
  AST_Expression::AST_ExprValue  *exprvalue;

  exprvalue = this->eval(AST_Expression::EK_const);
  switch (exprvalue->et) {
  case AST_Expression::EV_short: 	    // Expression value is short
    sprintf(o, "%d",exprvalue->u.sval);
    break;
  case AST_Expression::EV_ushort:	// Expression value is unsigned short
    sprintf(o, "%d",exprvalue->u.usval);
    break;
  case AST_Expression::EV_long:			// Expression value is long
    sprintf(o, "%ld",exprvalue->u.lval);
    break;
  case AST_Expression::EV_ulong:	// Expression value is unsigned long
    sprintf(o, "%ld",exprvalue->u.ulval);
    break;
  case AST_Expression::EV_longlong:	// Expression value is long long
    // I am going to map this to long ... there is no reference to a 
    // CORBA_long_long ...
    sprintf(o, "%ld",exprvalue->u.lval);
    break;
  case AST_Expression::EV_ulonglong:  // Expression value is unsigned long long
    // Dito as above
    sprintf(o, "%ld",exprvalue->u.ulval);
    break;
  case AST_Expression::EV_float:	// Expression value is 32-bit float
    sprintf(o, "%g",exprvalue->u.fval);
    break;
  case AST_Expression::EV_double:	// Expression value is 64-bit float
    sprintf(o, "%g",exprvalue->u.dval);
    break;
  case AST_Expression::EV_longdouble:	// Expression value is 128-bit float
    // nothing about this in the CORBA document I have. I will just map to
    // a 64-bit float, it is going to work for our pratical proposals.
    sprintf(o, "%g",exprvalue->u.dval);
    break;
  case AST_Expression::EV_char:		// Expression value is char
    sprintf(o, "%d",exprvalue->u.cval);
    break;
  case AST_Expression::EV_wchar:	// Expression value is wide char   
    // again, I don't know anything about this.
    sprintf(o, "%d",exprvalue->u.cval);
    break;
  case AST_Expression::EV_octet:	// Expression value is unsigned char
    sprintf(o, "%d",exprvalue->u.oval);
    break;
  case AST_Expression::EV_bool:		// Expression value is boolean
    sprintf(o, "%d",(int)exprvalue->u.bval);
    break;
  case AST_Expression::EV_string:	// Expression value is char *
    sprintf(o, "%lx",(long)exprvalue->u.strval);
    break;
  case AST_Expression::EV_wstring:	// Expression value is wchar_t *
    sprintf(o, "%lx",(long)exprvalue->u.strval);
    break;
  case AST_Expression::EV_any:		// Expression value is any of above
    // I don't see the meaning for constant, so I won't generate anything.
    // I started trying to generate (CORBA_any) blablabla, but there is
    // no value, there is no field in the struct exprvalue to get it from.
    break;
  case AST_Expression::EV_void:		// Expression value is void (absent)
    break;
  case AST_Expression::EV_none:		// Expression value is missing
    break;
  default: {
    // this is not expected to happen, since it would be an IDL error
    cout << "Error in  be_expr_val::ctranslation\n";
    BE_abort();
    }
  }
}

/*
 * BE_EnumVal
 */

be_enum_val::be_enum_val()
{
}
be_enum_val::be_enum_val(unsigned long v, UTL_ScopedName *n, UTL_StrList *p)
	   : AST_Constant(AST_Expression::EV_ulong,
			  AST_Decl::NT_enum_val,
			  new AST_Expression(v),
			  n,
			  p),
	     AST_Decl(AST_Decl::NT_enum_val, n, p)
{
}

void
be_enum_val::ctranslation(ostream &o)
{
  be_constant cte(this->et(), this->constant_value(), this->name(),
		  this->pragmas());
  cte.ctranslation(o);
}

// Narrowing
IMPL_NARROW_METHODS1(be_enum_val, AST_EnumVal)
IMPL_NARROW_FROM_DECL(be_enum_val)

/*
 * BE_Array
 */
be_array::be_array() : pd_already_defined(I_FALSE)
{
}
be_array::be_array(UTL_ScopedName *n, unsigned long ndims, UTL_ExprList
		   *dims)
  : pd_already_defined(I_FALSE),
    AST_Array(n, ndims, dims),
    AST_Decl(AST_Decl::NT_array, n, NULL)
{
}

idl_bool
be_array::already_defined()
{
  return pd_already_defined;
}

/* make_not_defined and make_defined are used in
 * translate_from_inherited_interfaces
 */ 

void
be_array::make_not_defined()
{
  pd_already_defined = I_FALSE;
}

void
be_array::make_defined()
{
  pd_already_defined = I_TRUE;
}

/*
 * translate_array_dimentions
 */
void
be_array::translate_array_dimensions (ostream &o, char before_dim,
			    char after_dim)
{
  // About before_dim, after_dim:
  // if it is defining the array type as
  // CORBA_ARRAY_..., it will be "_". If it
  // is creating the type with type, it will
  // be "["

  if (before_dim == '_')
    o << before_dim; 
  for (int i=0; i< n_dims(); i++){
    be_expr_val            expr(dims()[i], dims()[i]->ev()->et); 
    if (before_dim == '[')
      o << before_dim;        
    expr.value_only_ctranslation(o);
    o << after_dim;         
  }
}

/*
 * Ctranslation for Array
 */
void
be_array::ctranslation(ostream &o)
{
  // History of designing this function: in the begining I thought that
  // being an array, all there was to do was to translate the base type. 
  // The dimensions would be taken care in "be_typedef::ctranslation" or
  // translate_fields().
  // **BUT** actually arrays appear as building blocks for sequences, so
  // they need to have a typename associate with them early in the local
  // declarations. 
  char *typename_var = (char*)malloc(1);
  typename_var[0] = 0;
  char digits[6];
  digits[0] = '\0';
  char digits_part[255];
  char digits_list[1024];
  digits_part[0] = '\0';
  digits_list[0] = '\0';

  for (int i = 0; i< n_dims(); i++) {
    be_expr_val expr(dims()[i], dims()[i]->ev()->et);
    expr.value_only_2string_ctranslation(digits);
    strcat(digits_part, digits);
    strcat(digits_part, "_");
    strcat(digits_list, digits);
    strcat(digits_list, " ");
  }

  if (pd_already_defined || imported()) {
    // just output the name of the type 
    o << "CORBA_array_";
    typename_var = translate_typename_for_pure_name(base_type(), typename_var);
    o << typename_var;
    translate_array_dimensions(o, '_', '_');
    o << " ";
  } else {
    check_for_struct_array_base_type(base_type(), o);
    // define the name of the type
    o << "#ifndef _CORBA_array_";
    typename_var = translate_typename_for_pure_name(base_type(), typename_var);
    o << typename_var;
    translate_array_dimensions(o, '_', '_');
    o << "_defined\n";
    o << "#define _CORBA_array_";
    o << typename_var;
    translate_array_dimensions(o, '_', '_');
    o << "_defined\n";
    o << "typedef ";
    translate_type(base_type(), o);
    o << " CORBA_array_" << typename_var;
    translate_array_dimensions(o, '_', '_');
    translate_array_dimensions(o, '[', ']');
    o <<";\n#endif\n\n";

    // has to take care of local type nodes, ie, there is not a name for them
    if (strcmp(this->name()->last_component()->get_string(), 
	       "local type") == 0) {
      // change it
      char  *newlocaltype = (char *) malloc(strlen(typename_var)
					    +strlen("CORBA_array_")
					    + 6 * n_dims());
      if (newlocaltype == NULL) {
	cout << "ERROR in array translation: coult not allocate\n";
	BE_abort();
      }
      sprintf(newlocaltype, "CORBA_array_%s_",typename_var);
      for (int i = 0; i< n_dims(); i++) {
	be_expr_val expr(dims()[i], dims()[i]->ev()->et);
      
	expr.value_only_2string_ctranslation(digits);
	strcat(newlocaltype, digits);
	strcat(newlocaltype, "_");
      }
      this->set_name(new UTL_ScopedName(
			   new Identifier(newlocaltype, 1, 0, I_FALSE),
			   NULL));
    }
    else {
      o << "typedef CORBA_array_" << typename_var << "_" << digits << " ";
      translate_scoped_name ( (AST_Decl *) this, o);
      o << ";\n";
    }
  }

  if (!pd_already_defined) {
      // Let's send information about this complex type to the list that
      // is keeping track of them.
      build_type_info();
      pd_already_defined = I_TRUE;
  }
  free(typename_var);
}

void
be_array::build_type_info()
{
  char *typename_var = (char*)malloc(1);
  typename_var[0] = 0;
  char digits[6];
  digits[0] = '\0';
  char digits_part[255];
  char digits_list[1024];
  digits_part[0] = '\0';
  digits_list[0] = '\0';

  typename_var = translate_typename_for_pure_name(base_type(), typename_var);
  // has to take care of local type nodes, ie, there is not a name for them
  if (strcmp(this->name()->last_component()->get_string(), 
	     "local type") == 0) {
    // change it
    char  *newlocaltype = (char *) malloc(strlen(typename_var)
					  +strlen("CORBA_array_")
					  + 6 * n_dims());
    if (newlocaltype == NULL) {
      cout << "ERROR in array translation: coult not allocate\n";
      BE_abort();
    }
    sprintf(newlocaltype, "CORBA_array_%s_",typename_var);
    for (int i = 0; i< n_dims(); i++) {
      be_expr_val expr(dims()[i], dims()[i]->ev()->et);
      
      expr.value_only_2string_ctranslation(digits);
      strcat(newlocaltype, digits);
      strcat(newlocaltype, "_");
    }
    this->set_name(new UTL_ScopedName(
			new Identifier(newlocaltype, 1, 0, I_FALSE), NULL));
  }
  for (int i = 0; i< n_dims(); i++) {
    be_expr_val expr(dims()[i], dims()[i]->ev()->et);
    expr.value_only_2string_ctranslation(digits);
    strcat(digits_part, digits);
    strcat(digits_part, "_");
    strcat(digits_list, digits);
    strcat(digits_list, " ");
  }


    /* determine basic type for the array */
    AST_Type *tmp_type;
    int t;
    char *charcode;
    determine_specific_type(base_type(), &tmp_type, &t);
    generate_typecode_2string(t, &charcode);

    /* we need to translate the base type for the array. */
    char *basetypename_var;
    if (base_type()->node_type()  == AST_Decl::NT_pre_defined) {
      be_predefined_type *b;
      b = be_predefined_type::narrow_from_decl(base_type());
      if (b == NULL) {
	// error in narrowing
	assert(0);
      }
      else 
	basetypename_var = b->ctranslation2dstring();
    }
    else 
      basetypename_var = translate_scoped_name2dstring(base_type());

    char *tname = (char *)malloc(strlen("CORBA_array_")
				 +strlen(typename_var)+1+
				 strlen(digits_part)+1);
    // typename_var and digits are already properly set 
    assert(tname != NULL);
    char *tinfo = (char *)malloc(strlen("NT_Array NDIMS ")+6 + 
				 + strlen(digits_list) + 2 +
				 + strlen(" BASETYPENAME ") 
				 + strlen(basetypename_var) + 1
				 + strlen(" BASETYPECODE ") 
				 + strlen(charcode) + 1);
    assert(tinfo != NULL);
    sprintf(tinfo, " NT_array NDIMS %s ", digits_list);
    sprintf(tname, "CORBA_array_");
    strcat(tname, typename_var);
    strcat(tname, "_");
    strcat(tname, digits_part);
    strcat(tinfo, "BaseTypeName ");
    strcat(tinfo, basetypename_var);
    strcat(tinfo, " BASETYPECODE ");
    strcat(tinfo, charcode);
    be_description_type_list->add(tname, tinfo);
    free(basetypename_var);
}


// Narrowing
IMPL_NARROW_METHODS1(be_array, AST_Array)
IMPL_NARROW_FROM_DECL(be_array)

/*
 * BE_Sequence
 */
be_sequence::be_sequence() : pd_already_defined(I_FALSE)
{
    typename_var = NULL;
    basetypename_var = NULL;
}
be_sequence::be_sequence(AST_Expression *v, AST_Type *t)
	   : pd_already_defined(I_FALSE),
	     AST_Sequence(v, t),
    	     AST_Decl(AST_Decl::NT_sequence,
		      new UTL_ScopedName(
				new Identifier("sequence", 1, 0, I_FALSE),
				NULL),
		      NULL)
{
    typename_var = NULL;
    basetypename_var = NULL;
}

/* make_not_defined and make_defined are used in
 * translate_from_inherited_interfaces
 */ 

void
be_sequence::make_not_defined()
{
  pd_already_defined = I_FALSE;
}

void
be_sequence::make_defined()
{
  pd_already_defined = I_TRUE;
}

idl_bool
be_sequence::already_defined()
{
  return pd_already_defined;
}

void
be_sequence::set_types()
{
    if (typename_var != NULL) return;

    typename_var = (char *)malloc(1);
    basetypename_var = NULL;
    
    typename_var[0] = '\0';
    typename_var = translate_typename_for_pure_name(base_type(), typename_var);
    // Notice that the typename_var built with the call above holds
    // something like "long" if the base_type is a predefined type
    // (since it extracts the CORBA from the beginning). I will adjust
    // it in basetypename_var and use it as the type for _buffer and as
    // the basetypename information in the list of complex types being
    // kept. 
    AST_Type *specific_type;
    int node_type;
    determine_specific_type(base_type(), &specific_type, &node_type);
    switch(node_type) {
    case  AST_Decl::NT_pre_defined:
    case  AST_Decl::NT_string: {
      basetypename_var = (char *)malloc(strlen("CORBA_") + 
					strlen(typename_var) + 1);
      strcpy(basetypename_var, "CORBA_");
      strcat(basetypename_var, typename_var);
      if (node_type == AST_Decl::NT_string) {
	// we do not want the "maxlen" info in the name definition for
	// the string
	typename_var[6] = '\0';
      }
      break;
    }
    case  AST_Decl::NT_sequence: {
      be_sequence *bseq;
      bseq = be_sequence::narrow_from_decl(specific_type);
      if (bseq==NULL) { // narrow failed
	cout << "Error while narrowing sequence in "
	     << "be_sequence::ctranslation\n";
	BE_abort();
      };
      char digits[6];
      basetypename_var = strdup("CORBA_");
      basetypename_var = translate_typename_for_pure_name(bseq, basetypename_var);
      be_expr_val  expr(bseq->max_size(), 
			bseq->max_size()->ev()->et);
      expr.value_only_2string_ctranslation(digits); 
      basetypename_var = (char *) 
	realloc(basetypename_var, strlen(basetypename_var) +2 +strlen(digits));
      strcat(basetypename_var,"_"); 
      strcat(basetypename_var, digits);
      break;
      }
    default:
      basetypename_var = strdup(typename_var);
    }
}

/*
 * Ctranslation for BE_Sequence
 */
void
be_sequence::ctranslation(ostream &o)
{
  set_types();
  if (pd_already_defined || imported()) {
    // just using this definition. I first did like:
    // "translate_scoped_name ( (AST_Decl *) this, o)", but then I noticed
    // that in some cases the sequence type was defined (in terms of
    // CORBA_sequence_type_maximum, but not in terms of the name of the
    // type (which will be translated later, in the "declaration" part,
    // not in the "local declarations" part. So I will always output the
    // "real name of the type, and this will work fine both in the
    // "local declaration" and "declarations" case
    o << "CORBA_sequence_";
    o << typename_var << "_";
    be_expr_val expr(max_size(), max_size()->ev()->et);
    expr.value_only_ctranslation(o);
    o << " ";
  }

  if (!pd_already_defined) {
    check_for_struct_array_base_type(base_type(), o);
    char digits[6];
    be_expr_val  expr(max_size(), max_size()->ev()->et);
    expr.value_only_2string_ctranslation(digits);
    o << "#ifndef _CORBA_sequence_";
    o << typename_var << "_" << digits << "_defined\n"
      << "#define _CORBA_sequence_" << typename_var 
      << "_" << digits << "_defined\n";
    o << "typedef struct {\n"
      << "   CORBA_unsigned_long _maximum;\n"
      << "   CORBA_unsigned_long _length;\n"
      << "   " << basetypename_var << " *_buffer;\n"
      << "} CORBA_sequence_" << typename_var << "_" << digits << ";\n#endif\n";
    // has to take care of local type nodes, ie, there is not a name for them
    if (strcmp(this->name()->last_component()->get_string(), 
	       "local type") == 0) {
      // change it
      char  *newlocaltype = (char *) malloc(strlen(typename_var)
					   +strlen("CORBA_sequence_")
					   +strlen("999999"));
      if (newlocaltype == NULL) {
	cout << "ERROR in sequence translation: coult not allocate\n";
	BE_abort();
      }
      sprintf(newlocaltype, "CORBA_sequence_%s_%s",typename_var, digits);
      this->set_name(new UTL_ScopedName(
			   new Identifier(newlocaltype, 1, 0, I_FALSE),
			   NULL));
    }
    else {
      o << "typedef CORBA_sequence_" << typename_var << "_" << digits << " ";
      translate_scoped_name ( (AST_Decl *) this, o);
      o << ";\n";
    }


    build_type_info();
    pd_already_defined = I_TRUE;
  }
}

void
be_sequence::build_type_info()
{
    // Let's send information about this complex type to the list that
    // is keeping track of them.
    set_types();
    char *tname = (char *)malloc(strlen("CORBA_sequence_")
			   +strlen(typename_var)+6+1);  // 6 is maximum
    					         // size digit, 1 is for \0
    assert(tname != NULL);
    char digits[6];
    be_expr_val  expr(max_size(), max_size()->ev()->et);
    expr.value_only_2string_ctranslation(digits);

    if (strcmp(this->name()->last_component()->get_string(), 
	       "local type") == 0) {
      // change it
      char  *newlocaltype = (char *) malloc(strlen(typename_var)
					   +strlen("CORBA_sequence_")
					   +strlen("999999"));
      if (newlocaltype == NULL) {
	cout << "ERROR in sequence translation: coult not allocate\n";
	BE_abort();
      }
      sprintf(newlocaltype, "CORBA_sequence_%s_%s",typename_var, digits);
      this->set_name(new UTL_ScopedName(
			   new Identifier(newlocaltype, 1, 0, I_FALSE),
			   NULL));
    }
    /* determine basic type for the sequence */
    AST_Type *tmp_type;
    int t;
    char *charcode;
    determine_specific_type(base_type(), &tmp_type, &t);
    generate_typecode_2string(t, &charcode);
    char *tinfo = (char *)malloc(strlen("NT_Sequence MAXLEN ")+6 + 
				 + strlen(" BASETYPENAME ") 
				 + strlen(basetypename_var) + 1
				 + strlen(" BASETYPECODE ") 
				 + strlen(charcode) + 1);
    assert(tinfo != NULL);
    sprintf(tname, "CORBA_sequence_%s_%s",typename_var,digits);
    sprintf(tinfo, " NT_sequence MAXLEN %s BaseTypeName %s BASETYPECODE %s",
	    digits, basetypename_var, charcode); 
    be_description_type_list->add(tname, tinfo);
}    


// Narrowing
IMPL_NARROW_METHODS1(be_sequence, AST_Sequence)
IMPL_NARROW_FROM_DECL(be_sequence)

/*
 * BE_String
 */
be_string::be_string() : pd_already_defined(I_FALSE)
{
}
be_string::be_string(AST_Expression *v)
	 : pd_already_defined(I_FALSE),
	   AST_String(v),
	   AST_Decl(AST_Decl::NT_string,
		    new UTL_ScopedName(new Identifier("string", 1, 0, I_FALSE),
				       NULL),
		    NULL)
{
}

be_string::be_string(AST_Expression *v, long wide)
	 :  pd_already_defined(I_FALSE),
	    AST_String(v, wide),
	    AST_Decl(AST_Decl::NT_string,
		     wide == 1
		     ? new UTL_ScopedName(new Identifier("string",1,0,I_FALSE),
					  NULL)
		     : new UTL_ScopedName(new Identifier("wstring_t",
							 1,
							 0,
                                                        I_FALSE),
					  NULL),
		     NULL)
{
}

/* make_not_defined and make_defined are used in
 * translate_from_inherited_interfaces
 */ 

void
be_string::make_not_defined()
{
  pd_already_defined = I_FALSE;
}

void
be_string::make_defined()
{
  pd_already_defined = I_TRUE;
}



/*
 * translation to C for string declaration
 */
void
be_string::ctranslation(ostream &o)
{
  char digits[6];
  be_expr_val       expr(max_size(), max_size()->ev()->et);
  expr.value_only_2string_ctranslation(digits);

  if (pd_already_defined || imported()) {
    translate_scoped_name (this, o);
    o << " ";
  }
  if (!pd_already_defined) {
    // Now finally generates code for defining the string.
    o << "#ifndef _CORBA_string_";
    o << digits << "_defined\n"
      << "#define _CORBA_string_" 
      << digits
      << "_defined\n"
      << "typedef char *CORBA_string_"
      << digits
      << ";\n#endif\n";
    // has to take care of local type nodes, ie, there is not a name for them
    if (strcmp(this->name()->last_component()->get_string(), 
	       "local type") == 0) {
      // change it
      char  *newlocaltype = (char *) malloc(strlen("999999")  //a weird maxsize
					   +strlen("CORBA_string_"));
      if (newlocaltype == NULL) {
	cout << "ERROR in string translation: coult not allocate\n";
	BE_abort();
      }
      sprintf(newlocaltype, "CORBA_string_%s",digits);
      this->set_name(new UTL_ScopedName(
			   new Identifier(newlocaltype, 1, 0, I_FALSE),
			   NULL));
    }
    else {
      o << "typedef CORBA_string_" << digits << " ";
      translate_scoped_name ( (AST_Decl *) this, o);
      o << ";\n";
    }
    build_type_info();
    pd_already_defined = I_TRUE;
  }    
}

void
be_string::build_type_info()
{
    char digits[6];
    // Let's get information about MAXLENGTH
    be_expr_val       expr(max_size(), max_size()->ev()->et);
    expr.value_only_2string_ctranslation(digits);

    // Let's send information about this complex type to
    // the list that is keeping track of them.
    char *tname = (char *)malloc(strlen("CORBA_string_")+6+1);  // 6 is maximum
							// size digit, 1
							// is for \0
    assert(tname != NULL);
    char *tinfo = (char *)malloc(strlen(" NT_String MAXLEN ")+6+1);
    assert(tinfo != NULL);
    sprintf(tname, "CORBA_string_%s",digits);
    sprintf(tinfo, " NT_string MAXLEN %s", digits);
    be_description_type_list->add(tname, tinfo);
    
    // has to take care of local type nodes, ie, there is not a name for them
    if (strcmp(this->name()->last_component()->get_string(), 
	       "local type") == 0) {
      // change it
      char  *newlocaltype = (char *) malloc(strlen("999999")  //a weird maxsize
					   +strlen("CORBA_string_"));
      if (newlocaltype == NULL) {
	cout << "ERROR in string translation: coult not allocate\n";
	BE_abort();
      }
      sprintf(newlocaltype, "CORBA_string_%s",digits);
      this->set_name(new UTL_ScopedName(
			   new Identifier(newlocaltype, 1, 0, I_FALSE),
			   NULL));
    }
}    

// Narrowing
IMPL_NARROW_METHODS1(be_string, AST_String)
IMPL_NARROW_FROM_DECL(be_string)

be_typedef::be_typedef()
{
}
be_typedef::be_typedef(AST_Type *bt, UTL_ScopedName *n, UTL_StrList *p)
	  : AST_Typedef(bt, n, p),
	    AST_Decl(AST_Decl::NT_typedef, n, p)
{
}

/*
 * Ctranslation for typedef
 */
void
be_typedef::ctranslation(ostream &o)
{
    AST_Type *base_typ = base_type();
    if (!base_typ->imported()) {
	switch(base_typ->node_type()) {
	case AST_Decl::NT_sequence:
	{
	    be_sequence *bseq;
	    bseq = be_sequence::narrow_from_decl(base_typ);
	    if (!bseq->already_defined()) {
		base_typ->ctranslation(o);
	    }
	    break;
	}
	case AST_Decl::NT_array:
	{
	    be_array *barray;
	    barray = be_array::narrow_from_decl(base_typ);
	    if (!barray->already_defined()) {
		base_typ->ctranslation(o);
	    }
	    break;
	}
	case AST_Decl::NT_struct:
	{
	    be_structure *bstruct;
	    bstruct = be_structure::narrow_from_decl(base_typ);
	    if (!bstruct->already_defined()) {
		base_typ->ctranslation(o);
	    }
	    break;
	}
	default:
	    break;
	}
    }
  o << "typedef ";
  translate_type (base_type(), o);
  translate_scoped_name(this, o);
  o << ";\n";

  build_type_info();
}

void
be_typedef::build_type_info()
{
  // if the identifier being defined relates to a complex type,
  // we need to output information about this identifies to the
  // list of complex types defined.
  char *tmp;
  char *tname;
  tname = translate_scoped_name2dstring(this);
  switch (base_type()->node_type()) {
  case AST_Decl::NT_typedef: {
    char *tinfo; 
    tmp = translate_scoped_name2dstring(base_type());
    tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen(tmp)+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF %s", tmp);
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  }
  case AST_Decl::NT_enum: {
    char *tinfo; 
    tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen("NT_enum")+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF NT_enum");
    be_description_type_list->add(tname, tinfo);
    break;
  }
  case AST_Decl::NT_pre_defined:{
    be_predefined_type *be;
    be = be_predefined_type::narrow_from_decl(base_type());
    if (be==NULL) {
      printf("Narrowing to pre_defined_type in typedef::ctranslation failed\n");
      BE_abort();
    }
    tmp = be->ctranslation2dstring();
    char *tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen(tmp)+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF %s", tmp);
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  } 
  case AST_Decl::NT_string: {
    char *tinfo; 
    be_string *be;
    be = be_string::narrow_from_decl(base_type());
    if (be==NULL) {
      printf("Narrowing to string in typedef::ctranslation failed\n");
      BE_abort();
    }
    if (base_type()->imported()) {
      be->build_type_info();
    }
    tmp = translate_scoped_name2dstring(base_type());
    tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen(tmp)+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF %s", tmp);
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  }
  case AST_Decl::NT_sequence: {
    char *tinfo; 
    be_sequence *be;
    be = be_sequence::narrow_from_decl(base_type());
    if (be==NULL) {
      printf("Narrowing to sequence in typedef::ctranslation failed\n");
      BE_abort();
    }
    if (base_type()->imported()) {
      be->build_type_info();
    }
    tmp = translate_scoped_name2dstring(base_type());
    tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen(tmp)+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF %s", tmp);
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  }
  case AST_Decl::NT_array:{
    char *tinfo; 
    be_array *be;
    be = be_array::narrow_from_decl(base_type());
    if (be==NULL) {
      printf("Narrowing to array in typedef::ctranslation failed\n");
      BE_abort();
    }
    if (base_type()->imported()) {
      be->build_type_info();
    }
    tmp = translate_scoped_name2dstring(base_type());
    tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen(tmp)+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF %s", tmp);
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  }
  case AST_Decl::NT_struct:{
    char *tinfo; 
    be_structure *be;
    be = be_structure::narrow_from_decl(base_type());
    if (be==NULL) {
      printf("Narrowing to struct in typedef::ctranslation failed\n");
      BE_abort();
    }
    if (base_type()->imported()) {
      be->build_type_info();
    }
    tmp = translate_scoped_name2dstring(base_type());
    tinfo = (char*) malloc(strlen("TYPEDEF_OF ")+ strlen(tmp)+ 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF %s", tmp);
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  }
  case AST_Decl::NT_interface:{
    char *tinfo; 
    tinfo = (char*) malloc(strlen("TYPEDEF_OF CORBA_Object") + 1);
    assert(tinfo!=NULL);
    sprintf(tinfo, "TYPEDEF_OF CORBA_Object");
    be_description_type_list->add(tname, tinfo);
    free(tmp);
    break;
  }
  case AST_Decl::NT_union:
    // not dealing with this yet in a complete way, so let's forget
    // for now
    assert(0);
    break;
  default:
    assert(0);
  }   
}

// Narrowing
IMPL_NARROW_METHODS1(be_typedef, AST_Typedef)
IMPL_NARROW_FROM_DECL(be_typedef)



/* 
 * BE_Idlist
 */
be_idlist::be_idlist(UTL_IdList *l) 
  : UTL_IdList(l->head(), (UTL_IdList *) l->tail())
{
}

void
be_idlist::ctranslation(ostream &o)
{

  char *out;

  out = ctranslation2dstring();
  o << out;
  free(out);
}

char *
be_idlist::ctranslation2dstring()
{
  
  UTL_IdListActiveIterator	*i = 
                      new UTL_IdListActiveIterator((UTL_IdList *) this);
  long				first = I_TRUE;
  long				second = I_FALSE;
  char *o = (char *)malloc(2);

  strcpy(o, "");

  while (!(i->is_done())) {
    char *tmp;
    if (!first)
      strcat(o, "_");
    else if (second)
      first = second = I_FALSE;
    tmp = i->item()->get_string();
    o = (char *)realloc(o, strlen(o) + strlen(tmp) + 2);
    strcat(o, tmp);
    // I don't completely trust/understand the following part. 
    // It is based on the dump() code for UTL_IdList, and some experimentation
    // (therefore some changes)

    if (first) {
      if (strcmp(i->item()->get_string(), "") != 0)
	first = I_FALSE;
      else
	second = I_TRUE;
    }
    i->next();
  }
  delete i;
  // cout << "outside be_idlist:ctranslation2dstring\n";
  return o;
}

void
be_idlist::translate_scope_part(char *o)
{
  
  UTL_IdListActiveIterator	*i = 
                      new UTL_IdListActiveIterator((UTL_IdList *) this);
  long				first = I_TRUE;
  long				second = I_FALSE;
  long                          done = i->is_done();
  long                          nparts = tail()->length();
  strcpy(o, "");

  while (!done) {
    if (!first)
      strcat(o, "_");
    else if (second)
      first = second = I_FALSE;
    // I don't completely trust/understand the following part. 
    // It is based on the dump() code for UTL_IdList, and some experimentation
    // (therefore some changes)
    strcat(o, i->item()->get_string());
    if (first) {
      if (strcmp(i->item()->get_string(), "") != 0) {
	nparts--;
	first = I_FALSE;
      }
      else {
	second = I_TRUE;
      }
    }
    else
      nparts--;

    // check if the one before last
    if (nparts == 1)
      done = I_TRUE;
    else
      i->next();
  }
  delete i;
}
  
/*
void
be_idlist::translate4attribute(const char *acess, char *out)
{
  translate_scope_part(out);
  strcat(out, "__");
  strcat(out, access);
  strcat(out, last_component()->get_string());
}
*/
