/*
 * Copyright 1999 by Brown University Computer Science
 * 115 Waterman Street, 4th Floor, Providence, Rhode Island, 02906, U.S.A
 * All rights reserved.
 *
 * Permission is hereby granted for free use and modification of this
 * software.  However, this copyright must remain at all times.
 * Re-distribution is restricted to those with the express written consent
 * of the copyright holder.  Permission can be obtained by mailing
 * cs015headtas@cs.brown.edu.  
 */

package NGP;

/**
 * A subclass of <code>Timer</code> that will only activate a specific number
 * of times.
 *
 * @author Matt Chotin (<a href="mailto:mhc@cs.brown.edu">mhc</a>)
 */

public abstract class DefiniteTimer extends Timer {

  /** Times to execute and times already executed */
  private int _times, _timesExecuted;

  /**
   * Creates a <code>DefinteTimer</code> with a delay and a specific number
   * of times to {@link Timer#activate() activate}.
   *
   * @param delay milliseconds between activate calls
   * @param numTimes times to activate before stopping
   */
  public DefiniteTimer(int delay, int numTimes) {
    super(delay);
    _times = numTimes;
    _timesExecuted = 0;
  }

  /**
   * Normal users need not deal with this method!
   * <p>
   * Call the superclass actionPerformed, increment the number of times
   * executed, and check to see if we should stop.
   *
   * @param e the java.awt.event.ActionEvent that was sent, we do not deal
   * with it
   */
  public void actionPerformed(java.awt.event.ActionEvent e) {
    super.actionPerformed(e);
    if (++_timesExecuted == _times) this.stop();
  }

  /**
   * Reset the number of times executed back to 0 and start the Timer.
   */
  public void start() {
    _timesExecuted = 0;
    super.start();
  }

  /**
   * Set the number of times this Timer should call
   * {@link Timer#activate() activate}.
   *
   * @param t times to call activate
   * @see #getTimesToExecute() getTimesToExecute
   * @see #getTimesAlreadyExecuted() getTimesAlreadyExecuted
   */
  public void setTimesToExecute(int t) {
    _times = t;
  }

  /**
   * Return the number of times this Timer will
   * {@link Timer#activate() activate}.
   *
   * @return the number of times activate will be called
   * @see #setTimesToExecute(int) setTimesToExecute
   * @see #getTimesAlreadyExecuted() getTimesAlreadyExecuted
   */
  public int getTimesToExecute() {
    return _times;
  }

  /**
   * Return how many times {@link Timer#activate() activate}
   * has already been called.
   *
   * @return how many times activate has already been called
   * @see #getTimesToExecute() getTimesToExecute
   * @see #setTimesToExecute(int) setTimesToExecute
   */
  public int getTimesAlreadyExecuted() {
    return _timesExecuted;
  }

}

