/*
 * Copyright 1999 by Brown University Computer Science
 * 115 Waterman Street, 4th Floor, Providence, Rhode Island, 02906, U.S.A
 * All rights reserved.
 *
 * Permission is hereby granted for free use and modification of this
 * software.  However, this copyright must remain at all times.
 * Re-distribution is restricted to those with the express written consent
 * of the copyright holder.  Permission can be obtained by mailing
 * cs015headtas@cs.brown.edu.  
 */

package NGP.Components;

//an import statement means I can refer to a class as if I specified the
//whole package
import javax.swing.JTextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * A text box for single-line text entry.  It is a subclass of the
 * Swing JTextField.  You can change the text alignment easily and when
 * Return is pressed the {@link #returnPressed() returnPressed} method is
 * called.
 *
 * @see <a href="http://java.sun.com/products/jdk/1.2/docs/api/javax/swing/JTextField.html">javax.swing.JTextField</a>
 *
 * @author Matt Chotin (<a href="mailto:mhc@cs.brown.edu">mhc</a>)
 */

public class TextBox extends JTextField implements NGP.Component {

  /**
   * Construct a text box in container.
   *
   * @param container the graphical container for the text box
   */
  public TextBox(NGP.Container container) {
    super();
    this.addActionListener(new TextListener());
    container.add(this);
  }


  /**
   * Construct the text box in container with default text.
   *
   * @param container the graphical container for the text box
   * @param text the default text for the text box
   */
  public TextBox(NGP.Container container, String text) {
    super(text);
    this.addActionListener(new TextListener());
    container.add(this);
  }

  /**
   * Construct the text box in container with specified width.
   *
   * @param container the graphical container for the text box
   * @param cols the number of columns (characters) a text box should take
   */
  public TextBox(NGP.Container container, int cols) {
    super(cols);
    this.addActionListener(new TextListener());
    container.add(this);
  }

  /**
   * Construct the text box in container with default text and specified width.
   *
   * @param container the graphical container for the text box
   * @param text the default text for the text box
   * @param cols the number of columns (characters) a text box should take
   */
  public TextBox(NGP.Container container, String text, int cols) {
    super(text, cols);
    this.addActionListener(new TextListener());
    container.add(this);
  }

  /**
   * Align the text in the box to the left.
   */
  public void alignLeft() {
    this.setHorizontalAlignment(JTextField.LEFT);
  }

  /**
   * Align the text in the box to the right.
   */
  public void alignRight() {
    this.setHorizontalAlignment(JTextField.RIGHT);
  }

  /**
   * Align the text in the box to the center.
   */
  public void alignCenter() {
    this.setHorizontalAlignment(JTextField.CENTER);
  }

  /**
   * Set the preferred dimensions of this Component.
   *
   * @param d the <code>Dimension</code> for this <code>Component</code>.
   * @see #getDimension() getDimension
   */
  public void setDimension(java.awt.Dimension d) {
    this.setPreferredSize(d);
  }

  /**
   * Get the preferred dimensions of this Component.
   *
   * @return the preferred <code>Dimension</code> of this
   * <code>Component</code>.
   * @see #setDimension(java.awt.Dimension) setDimension
   */
  public java.awt.Dimension getDimension() {
    return this.getPreferredSize();
  }

  /**
   * This will be called when Return is pressed with focus in the text box.
   * If you want it to do something you must override it.
   */
  public void returnPressed() { }

  /**
   * Convert the string value of the text box to an int. This is useful for
   * sending the text informationto something that is expecting an integer.
   *
   * @return int representation of text value (0 if string not recognized as
   * int)
   */
  public final int textToInt() {
    try {
      return Integer.valueOf(this.getText()).intValue();
    }
    catch (java.lang.NumberFormatException e) {}
    return 0;
  }

  /**
   * Convert the string value of the text box to a double.  This is useful
   * for sending the text information to something that is expecting an double.
   *
   * @return double representation of text value (0 if string not recognized as
   * double)
   */
  public final double textToDouble() {
    try {
      return Double.valueOf(this.getText()).doubleValue();
    }
    catch (java.lang.NumberFormatException e) {}
    return 0;
  }

  /**
   * This inner-class is a listener for the text box.  It will detect when the
   * user presses Return and call the {@link #returnPressed() returnPressed}
   * method of the text box.
   */
  private class TextListener implements ActionListener {
    public void actionPerformed(ActionEvent e) {
      returnPressed();
    }
  }

}
