/*************************************************************************/
/*************************************************************************/
/*

plan11.c
Back to trying to pass the large integers...

plan10.c
I got tired of trying to get it to pass large integers...  so I hard 
coded them...  Also erased memory hog plan[5000]

plan9.c
A copy of plan7, not plan8.  This version attempts to load the lookup 
table onto the 332 near the inverted position and use it there.  This is 
b/c plan8 has shown that the sun lags too much to be used for the 
inverted position, even when not using the lookup table.

Also implements a loading fault indicator (LEDs)

plan7.c
Sends the motor command with the rest of the info since the sun lags.  
Better for data analysis purposes.

plan6.c 
Sends the sun data, stops sending, waits for a reply, sends more data... etc.

plan4.c
processes time, position, and velocity on board and sends it to the sun

plan3.c
Gets a command to send a packet, sends the packet, stops sending data, 
gets a command, then waits for a command to send another packet.

plan2.c 
Same as plan.c, but incorporates the fixed isr period timing so that it 
is really 10 ms

plan.c
stripped down version of previous programs - only basic functionality 
included

Automatically turns the servo on and enables the slave mode.

*/
/*************************************************************************/
/*************************************************************************/

/* System include files */
#include "interrupt.h"
#include "sys.h"
#include "TPU.h"

/*************************************************************************/

#define TRUE	1
#define FALSE	0
#define NEGATIVE 1
#define POSITIVE 2
#define ZERO 0
#define ERROR_LIGHT_ON 4
#define ERROR_LIGHT_OFF 0
#define LOAD_LIGHT_ON 8
#define LOAD_LIGHT_OFF 0
#define CAPTURE_LIGHT_ON 16
#define CAPTURE_LIGHT_OFF 0

/*************************************************************************/
/*************************************************************************/
/*************************************************************************/

/* Clock stuff */

#define PITR ((volatile unsigned short *) 0xfffa24)
#define PICR ((volatile unsigned short *) 0xfffa22)
#define PIT_VECTOR 0x44
#define PIT_LEVEL 6

#define INTERRUPT_PERIOD 10  /* ms */
#define COUNTS_PER_1MS 2103  /* number of clock units per 1 ms */
/*
#define N_POS_C 30
#define N_VEL_C 30
#define MIN_POS_ENCODER_C 900
#define MAX_POS_ENCODER_C 1100
#define MIN_VEL_ENCODER_C -1000
#define MAX_VEL_ENCODER_C 1000
*/
/*************************************************************************/

int the_time = 0;
int servo_on = 0;
int send_packets = 0;
int period_length;
int error_light;
int capture_light;
int load_light;

/* pseudo constants */
int N_POS_C;
int N_VEL_C;
int MIN_POS_ENCODER_C;
int MAX_POS_ENCODER_C;
int MIN_VEL_ENCODER_C;
int MAX_VEL_ENCODER_C;


/* arrays */
int command_table[110][40];

/* structures */
typedef struct STATE_STRUCT {
        int pos;
        int vel;
}
STATE;

typedef struct INDICES_STRUCT {
        int pos;
        int vel;
}
INDICES;

/*************************************************************************/

extern void pit_isr_asm();

asm(" 
  .globl _pit_isr_asm

_pit_isr_asm:
/*    orb #0x2,(0xfffa11)    /* set bit 1 of porte */
   link a6,#0
   moveml #0xc0c0,sp@-
   jsr _pit_isr
/* The quick version just increments the time
   movel _the_time,d0
   addql #1,d0
   movel d0,_the_time
*/
   moveml sp@+,#0x4303
/*    andb #0xfd,(0xfffa11)  /* reset bit 1 of porte */
   rte
");

/*************************************************************************/

initialize_pit( period )
     int period;
{

  *PITR = 0;
  install_exception_handler( PIT_VECTOR, pit_isr_asm );
  *PICR = ((PIT_LEVEL&0x7)<<8) + (PIT_VECTOR&0xff);
  set_pit_period( period );
}

/*************************************************************************/

/* See page 4-19 in 68332 manual */
set_pit_period( microseconds )
     int microseconds;
{
  int pitm;

  if ( microseconds < 62000 )
    {
      pitm = microseconds >> 8;
      if ( pitm <= 0 )
	pitm = 1;
      if ( pitm > 0xff )
	pitm = 0xff;
    }
  else
    {
      pitm = microseconds >> 16;
      if ( pitm <= 0 )
	pitm = 1;
      if ( pitm > 0xff )
	pitm = 0xff;
      pitm += 0x100;
    }
  *PITR = pitm;
}

/*************************************************************************/
/*************************************************************************/
/* Encoder stuff */

#define CH1_A 0
#define CH1_B 1
#define CH2_A 2
#define CH2_B 3

#define QDEC_TPU_MODE 6

#define TPU_PARAMETER_RAM ((volatile unsigned short *) 0xffff00)

/* Location of parameters */
#define EDGE_TIME 0 /* value of TCR1 at transition */
#define POSITION_COUNT 1 /* counter for encoder counts */
#define TCR1_VALUE 2  /* access to current value of TCR1 */
#define CHAN_PINSTATE 3  /* signal value after last transition */
#define CORR_PINSTATE_ADDR 4 
   /* Need to set this to address of other channel's CHAN_PINSTATE */
#define EDGE_TIME_LSB_ADDR 5
   /* Need to set this to address of other channel's EDGE_TIME */


qdec_init( ch0, ch1 )
     int ch0, ch1;
{
  /* clear interrupt enable to prevent spurious interrupt */
  stuff_cier(ch0,0 );
  stuff_cier(ch1,0 );

  /* Disable channels by clearning the two channel priority bits */
  stuff_cpr( ch0, 0 );
  stuff_cpr( ch1, 0 );

  /* select qdec function */
  stuff_cfs( ch0, QDEC_TPU_MODE );
  stuff_cfs( ch1, QDEC_TPU_MODE );

  /* initialize pointers in parameter ram */
  TPU_PARAMETER_RAM[ ch0*8 + CORR_PINSTATE_ADDR ] = ch1*16 + CHAN_PINSTATE*2;
  TPU_PARAMETER_RAM[ ch1*8 + CORR_PINSTATE_ADDR ] = ch0*16 + CHAN_PINSTATE*2;
  TPU_PARAMETER_RAM[ ch0*8 + EDGE_TIME_LSB_ADDR ] = ch0*16 + 1;
  TPU_PARAMETER_RAM[ ch1*8 + EDGE_TIME_LSB_ADDR ] = ch0*16 + 1;

  /* initialize position count */
  TPU_PARAMETER_RAM[ ch0*8 + POSITION_COUNT ] = 0;

  /* select one channel as primary and one channel as secondary. */
  stuff_hsf( ch0, 0 );
  stuff_hsf( ch1, 1 );

  /* send initialize command to host service request register */
  stuff_hsrf( ch0, 3 );
  stuff_hsrf( ch1, 3 );

  /* enable channels by setting the two channel priority bits */
  stuff_cpr( ch0, 3 );
  stuff_cpr( ch1, 3 );
}


/*************************************************************************/
/*************************************************************************/
/* I am relying on the compiler's rules about sign extension to make this
work. (unsigned short) -> (short) -> (int) */

volatile int last_encoder1, last_encoder2;
volatile int pos1, pos2;
volatile int vel1, vel2;
volatile int last_pos1, last_pos2;
volatile int max_vel1, min_vel1, max_vel2, min_vel2;

/*new global vars for spin1*/
volatile int send_cw, send_ccw, auto_on, cw_move_done, ccw_move_done;
volatile int invert_on, max_pump_speed, too_fast_on_top, plan_step;
volatile int motor_command;
volatile int original_on;

read_positions()
{
  short one, two;

  one = (short) TPU_PARAMETER_RAM[ CH1_A*8 + POSITION_COUNT ];
  two = (short) TPU_PARAMETER_RAM[ CH2_A*8 + POSITION_COUNT ];
  pos1 = fix_rollover( one, last_encoder1, pos1 );
  pos2 = fix_rollover( two, last_encoder2, pos2 );
  last_encoder1 = one;
  last_encoder2 = two;
}

/*************************************************************************/
/* Okay, the goal here is to get rid of encoder rollover */

fix_rollover( new, previous, pos )
     int new, previous, pos;
{
  int increment;

  increment = new - previous;
  if ( increment > 32768 )
    increment -= 65536;
  if ( increment < -32768 )
    increment += 65536;

  return pos + increment;
}

/*************************************************************************/
/*************************************************************************/
int get_period_length( length )
        int length;   
{
/* this function changes time in ms to the units necessary to get an
interrupt of that many ms */

  int units;

  units = length * COUNTS_PER_1MS;
  printf( "units: %d\n", units );

  return( units );

}

/*************************************************************************/
/*************************************************************************/

init_everything()
{
  /* Make sure servo off */
  servo_on = 0;

  /* TPU initialization stuff. (you shouldn't need to understand this) */
  initialize_tpu_speeds( psck_div4, div1, div1 );

  /* Initialize encoders */
  qdec_init( CH1_A, CH1_B );
  qdec_init( CH2_A, CH2_B );
  last_encoder1 = 0;
  last_encoder2 = 0;

  /* Zero position related variables */
  pos1 = last_pos1 = 0;
  pos2 = last_pos2 = 0;
  vel1 = min_vel1 = max_vel1 = 0;
  vel2 = min_vel2 = max_vel2 = 0;

  /* PORTE and HARDWARE DEBUGGING STUFF */
  set_porte_output( 0xff );
  set_portf_output( 0xff );

  /* Set up the clock. 10000 -> interrupt roughly every 10ms */
  /* This is bullshit.  10000 is not equal to 10 ms*/
  /* get_period_length does the conversion from ms to clock units */
  period_length = get_period_length( INTERRUPT_PERIOD );
  printf( "period_length: %d\n", period_length );
  initialize_pit( period_length );

  motor_command = ZERO;
  send_packets = FALSE;
  error_light = ERROR_LIGHT_OFF;
  capture_light = CAPTURE_LIGHT_OFF;
  load_light = LOAD_LIGHT_OFF;
}

/*************************************************************************/
/*************************************************************************/
main()  {

  init_everything();

  printf( "Turning servo on.\n" );
  servo_on = TRUE;

  printf( "Zeroing encoders.\n" );
  TPU_PARAMETER_RAM[ CH1_A*8 + POSITION_COUNT ] = 0;
  TPU_PARAMETER_RAM[ CH2_A*8 + POSITION_COUNT ] = 0;
  last_encoder1 = 0;
  last_encoder2 = 0;
  /* Zero position related variables */
  pos1 = last_pos1 = 0;
  pos2 = last_pos2 = 0;
  vel1 = min_vel1 = max_vel1 = 0;
  vel2 = min_vel2 = max_vel2 = 0;

  printf( "Now in slave mode.\n" );
  while (1)
  {
	read_packets();
  }

}

/*************************************************************************/
/*************************************************************************/
load_lookup_table()  {

  int pos_index, vel_index;
  int handshake;
  
  load_light = LOAD_LIGHT_ON;

  /* receive constants */
  while( ( handshake = (getchar() & 0xff) ) != 0xff )
      error_light = ERROR_LIGHT_ON;
  N_POS_C = read_int_32();
  send_int_32( N_POS_C );

  while( ( handshake = (getchar() & 0xff) ) != 0xff )
      error_light = ERROR_LIGHT_ON;
  N_VEL_C = read_int_32();
  send_int_32( N_VEL_C );

  while( ( handshake = (getchar() & 0xff) ) != 0xff )
      error_light = ERROR_LIGHT_ON;
  MIN_POS_ENCODER_C = read_int_32();
  send_int_32( MIN_POS_ENCODER_C );

  while( ( handshake = (getchar() & 0xff) ) != 0xff )
      error_light = ERROR_LIGHT_ON;
  MAX_POS_ENCODER_C = read_int_32();
  send_int_32( MAX_POS_ENCODER_C );

  while( ( handshake = (getchar() & 0xff) ) != 0xff )
      error_light = ERROR_LIGHT_ON;
  MIN_VEL_ENCODER_C = read_int_32();
  send_int_32( MIN_VEL_ENCODER_C );

  while( ( handshake = (getchar() & 0xff) ) != 0xff )
      error_light = ERROR_LIGHT_ON;
  MAX_VEL_ENCODER_C = read_int_32();
  send_int_32( MAX_VEL_ENCODER_C );

  for ( pos_index = 0; pos_index < N_POS_C; pos_index++ )  {
	for ( vel_index = 0; vel_index < N_VEL_C; vel_index++ )  {
		while( ( handshake = (getchar() & 0xff) ) != 0xff )
		      error_light = ERROR_LIGHT_ON;
		command_table[pos_index][vel_index] = getchar();
		send_int_32( command_table[pos_index][vel_index] );
	}
  }

  load_light = LOAD_LIGHT_OFF;

}

/*************************************************************************/
/*************************************************************************/
int in_capture_range()  {

  if ( ( pos1 > MIN_POS_ENCODER_C ) && ( pos1 < MAX_POS_ENCODER_C ) )  {
        return( TRUE );
  }
  else  {
        return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
INDICES get_indices ( state )
        STATE state;
{
/* get the indices in the table of the current state (given in encoder
counts) */

  int pos_encoder, pos_index;
  int vel_encoder, vel_index;
  INDICES indices;

  pos_encoder = state.pos;
  pos_index = ( pos_encoder - MIN_POS_ENCODER_C );
  pos_index = pos_index * N_POS_C;
  pos_index = pos_index / ( MAX_POS_ENCODER_C - MIN_POS_ENCODER_C );
  indices.pos = pos_index;

  vel_encoder = state.vel;
  vel_index = ( vel_encoder - MIN_VEL_ENCODER_C );
  vel_index = vel_index * N_VEL_C;
  vel_index = vel_index / (MAX_VEL_ENCODER_C - MIN_VEL_ENCODER_C);
  indices.vel = vel_index;

  return( indices );
 
} 

/*************************************************************************/
/*************************************************************************/
int capture_command( indices )
	INDICES indices;
{
  int table_command;

  if ( command_table[indices.pos][indices.vel] == -1 )
	table_command = 0;
  else
	table_command = command_table[indices.pos][indices.vel];
  return( table_command );
 
}

/****************************************************************************/
/****************************************************************************/
int in_table( indices )
        INDICES indices;
{
/* checks to see if the current state is in the table */

  if ( indices.pos < N_POS_C && indices.pos > -1
        && indices.vel < N_VEL_C && indices.vel > -1 )
        return( TRUE );
  else
        return( FALSE );

}

/*************************************************************************/
/*************************************************************************/

/* this is an interrupt service routine.  DO NOT put any printf's in here 
or the interrupt servicing will take too long, and the machine will crash */
pit_isr()
{
  int delta1, delta2;
  STATE state;
  INDICES indices;

/* oscilliscope - to measure how much of the isr is used */
  write_portf( 0xff );

  the_time += 10;	/* keep the time in ms */

  read_positions();

  /* calculate velocities in encoder counts/second */
  delta1 = (pos1 - last_pos1);
  delta2 = (pos2 - last_pos2);
  vel1 = (pos1 - last_pos1)*100;
  vel2 = (pos2 - last_pos2)*100;
  /* *100 due to 10ms clock period */

  last_pos1 = pos1;
  last_pos2 = pos2;

  if ( servo_on )  {
	if ( in_capture_range() )  {
		capture_light = CAPTURE_LIGHT_ON;
		state.pos = pos1;
		state.vel = vel1;
		indices = get_indices( state );
		if ( in_table( indices ) )
			motor_command = capture_command( indices );
	}
	else
		capture_light = CAPTURE_LIGHT_OFF;
	write_porte( motor_command + capture_light + error_light
		 + load_light );
  }
  else  {
	write_porte( ZERO + error_light + capture_light + load_light );
  }

  if ( send_packets )  {
	/* handshake */
	putchar( 0xff );
	/* random packet_id */
	putchar( 1 );
	/* get data */
	send_int_32( the_time );
	send_int_32( pos1 );
	send_int_32( vel1 );
	send_int_32( motor_command );

	/* reset so as not to clog the sun */
	send_packets = FALSE;
  }

  /*oscilliscope - turn it off*/
  write_portf( 0 );

}

/*************************************************************************/
/*************************************************************************/

send_int_32( value )
     int value;
{
  putchar( 0xff & value );
  value >>= 8;
  putchar( 0xff & value );
  value >>= 8;
  putchar( 0xff & value );
  value >>= 8;
  putchar( 0xff & value );
}

/*************************************************************************/
/*************************************************************************/
int read_int_32()  {

  int i;
  int line;

  i = 0;
  line = getchar();
  i += (unsigned) ( line & 0xff );
  line = getchar();
  i += (unsigned) ( ( line & 0xff ) << 8 );
  line = getchar();
  i += (unsigned) ( ( line & 0xff ) << 16 );
  line = getchar();
  i += ( ( line & 0xff ) << 24 );

  return i;

}

/*************************************************************************/
/*************************************************************************/

send_int_16( value )
     int value;
{
  putchar( 0xff & value );
  value >>= 8;
  putchar( 0xff & value );
}

/*************************************************************************/
/*************************************************************************/

int errors;

read_packets()
{
  int value;
  int packet_id;

 while ( 1 )  {

  while( ( value = (getchar() & 0xff) ) != 0xff )
    {
      errors++;
      /* printf( "1: errors: %d; value: %d\n", errors, value ); */
    }
  packet_id = getchar();

  switch (packet_id)  {
	case 1:  /* go nowhere */
		motor_command = ZERO;
		send_packets = TRUE;
	break;
	case 2:  /* go positive*/
		motor_command = POSITIVE;
		send_packets = TRUE;
	break;
	case 3:  /* go negative */
		motor_command = NEGATIVE;
		send_packets = TRUE;
	break;
	case 4:  /* send data */
		the_time = -10;  /* initializes so that is 0 when sent */
		send_packets = TRUE;
	break;
	case 5:  /* stop sending data */
		send_packets = FALSE;
	break;
	case 6:  /* zero encoders */
  		TPU_PARAMETER_RAM[ CH1_A*8 + POSITION_COUNT ] = 0;
  		TPU_PARAMETER_RAM[ CH2_A*8 + POSITION_COUNT ] = 0;
  		last_encoder1 = 0;
  		last_encoder2 = 0;
  		/* Zero position related variables */
  		pos1 = last_pos1 = 0;
  		pos2 = last_pos2 = 0;
  		vel1 = min_vel1 = max_vel1 = 0;
  		vel2 = min_vel2 = max_vel2 = 0;
	break;
	case 7:  /* load lookup table */
		error_light = ERROR_LIGHT_OFF;
		load_lookup_table();
	break;
	default:
		return;
	break;
  }
 }		
}

/*************************************************************************/



