/****************************************************************************/

/* 

sim9.c
A copy of sim8, but this time I am including the roll-over functions to keep it in 
the table

sim8.c
A copy of sim7, but I am limiting the size of the queue so that the 
machine doesn't crash....

sim7.c
A copy of sim1, but I am fixing the motor torque miscalculation.

sim1.c
OK, so now we are in the two arm (acrobot) version.  This should 
generalize for the most part, but of course, there will be bugs...

This program simulates the pendulum and finds a path from start 
to goal by emulating a 3 branched tree and doing a breadth first 
search using a queue.  This search builds a lookup table indexed by 
position and velocity and contains the command for that state.

This improved version of sim3 writes to a file every now and then (every 
10K visited).  It also writes a depth file which keeps track of the 
current depth.

Also fixes the 2000 counts per rev problem...

This sim5 version fixes some problems with trying to get data from an 
empty queue, removes the zero command (makes WAY faster), and some other 
logical programming errors

debug4 works for forward time...
makes a difference in which one gets considered first

debug5 works for backwards time
tries to establish no priority for command

debug6 tries to get the zero command working too

debug7 tries to establish equal precedence for the commands

debug8 sets up a real model for the pendulum instead of guesses and hacks

debug9 is the cleaned up version, which will be kept if it still works....

Renamed to sim6 since it worked... :)

Also realized that precedence is still given to the earlier commands since 
it does not look at the entire level of the table when it allows them to 
have equal precedence...  on to sim7

Now we are at sim7 - major changes in logic and structure:

First, the next states are calculated.  Then, when the whole level of 
the tree is finished, the algorithm looks at the first one.  If it is in 
the same cell as it's parent, then it lets it progress on it's path 
(creates 3 new nodes).  If it is in a different cell and that cell has 
already been assigned on the table, then the path is pruned.  If it is 
in a cell that is nto occupied on the table, then the rest of the level 
is searched to see if there is a state present which is closer to the 
center of that cell than the current_state.  If a closer cell is found, 
then the path is allowed to continue but the command is not entered in 
the table.  If the current_state is indeed the one closest to the center 
of the cell, then the table is marked with that command and the path is 
allowed to continue.  Also note that the paths that don't go anywhere 
are also cut off at a certain number of dwell cycles.

sim8
No longer looks for the start state to stop at, just goes for a certain 
number of tree nodes.  Also writes a file that contains all the constants 
(command.info) so that plan-sun can upload it and use it.

  */

/****************************************************************************/
/* Include files */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>

/****************************************************************************/
/****************************************************************************/
/* Constants */

#define TRUE 1
#define FALSE 0
#define LEVEL_MARKER -100
#define N_POS_1 25
#define N_VEL_1 25
#define N_POS_2 25
#define N_VEL_2 25
#define UNOCCUPIED -1
#define START_POS_1 0.0  /* radians */
#define START_VEL_1 0.0
#define GOAL_POS_1 3.1416  /* radians */
#define GOAL_VEL_1 0.00
#define START_POS_2 0.0  /* radians */
#define START_VEL_2 0.0
#define GOAL_POS_2 0.0  /* radians */
#define GOAL_VEL_2 0.00
#define COUNTS_PER_REV 2000
#define PI 3.1416
#define MIN_POS_ENCODER_1 -2000  /* encoder counts */
#define MAX_POS_ENCODER_1 1999
#define MIN_VEL_ENCODER_1 -50000  /* encoder counts */
#define MAX_VEL_ENCODER_1 49999
#define MIN_POS_ENCODER_2 -2000  /* encoder counts */
#define MAX_POS_ENCODER_2 1999
#define MIN_VEL_ENCODER_2 -50000  /* encoder counts */
#define MAX_VEL_ENCODER_2 49999
#define DELTA_TIME -.030  /* seconds */
#define A 17.64  /* all empirical and not used right now */
#define B -65.99
#define C -0.1674
#define D -0.1674
#define E -0.1674
#define F -0.1674

#define FILE_SAVE_INTERVAL 50000
#define INFO_INTERVAL 1000
#define MAX_STATE_DWELL 5000
#define MAX_QUEUE_SIZE 25000

/* Global vars */

/* arrays */
int com_array[N_POS_1][N_VEL_1][N_POS_2][N_VEL_2];
int counts[N_POS_1][N_VEL_1][N_POS_2][N_VEL_2];

/* structures */

typedef struct STATE_STRUCT {
	float pos_1;
	float vel_1;
	float pos_2;
	float vel_2;
}
STATE;

typedef struct INDICES_STRUCT {
	int pos_1;
	int vel_1;
	int pos_2;
	int vel_2;
} 
INDICES; 

typedef struct INFO_STRUCT {
	int same;  /* as last indices: TRUE, FALSE, or LEVEL_MARKER */
	int command;
} 
INFO; 

/* data structures and definitions for the queue */

typedef struct QUEUE_NODE_STRUCT {
	STATE data;
	INFO info;
	struct QUEUE_NODE_STRUCT *next;
}
QUEUE_NODE; 

/* global b/c I have no freakin clue how to do it otherwise */
QUEUE_NODE *free_queue;
/* This is fine. CGA */
/* Thanks.  -Sean */

/****************************************************************************/
/****************************************************************************/
QUEUE_NODE * add_to_top( queue, state, info )
  QUEUE_NODE *queue;
  STATE state;
  INFO info;
{
  QUEUE_NODE *new_queue;

  /* allocate memory */
  new_queue = ( QUEUE_NODE * ) malloc( sizeof( QUEUE_NODE ) );
  if ( new_queue == NULL )  {
	printf ( "Memory allocation error" );                
	exit(-1);
  }                      

  new_queue->data = state;
  new_queue->next = queue;
  new_queue->info = info;

  return( new_queue ); 

}
/****************************************************************************/
/****************************************************************************/
QUEUE_NODE * pop_off_top( queue )
	QUEUE_NODE *queue;
{
  QUEUE_NODE *temp;

/* pops off the top of the queue and adds to the free_queue */

  if ( queue != NULL ) { 
	temp = free_queue;
	free_queue = queue;
	queue = queue->next;
	free_queue->next = temp;
  }
  else  {
  }

  return( queue );

}
/****************************************************************************/
/****************************************************************************/
QUEUE_NODE * add_to_end( queue, state, info )
	QUEUE_NODE *queue;
	STATE state;
	INFO info;
{

/* adds a node to the end of the queue by taking it from the free_queue 
if available or allocating space otherwise */

  if ( queue != NULL )  {
	queue->next = add_to_end( queue->next, state, info );
  }
  else if ( free_queue != NULL )  {
	queue = free_queue;
	free_queue = free_queue->next;
	queue->data = state;
	queue->info = info;
	queue->next = NULL;
  }
  else  {
	/* allocate memory */
	queue = ( QUEUE_NODE * ) malloc( sizeof( QUEUE_NODE ) );
	if ( queue == NULL )  {
		printf ( "Memory allocation error" );                
		exit(-1);
	}
	queue->data = state;
	queue->info = info;
	queue->next = NULL;
  }	

  return ( queue );

}	

/****************************************************************************/
/****************************************************************************/
STATE get_top_data( queue )
        QUEUE_NODE *queue;
{
  if ( queue != NULL )  {
	return( queue->data );
  }
  else  {
	printf( "queue is empty.\n" );
	exit ( -1 );
  }	

}                   
                   
/****************************************************************************/
/****************************************************************************/
INFO get_top_info( queue )
        QUEUE_NODE *queue;
{
  if ( queue != NULL )  {
	return( queue->info );
  }
  else  {
	printf( "queue is empty.\n" );
	exit ( -1 );
  }	

}                   
                   
/****************************************************************************/
/****************************************************************************/
init_array()  {

  int index1, index2, index3, index4;

  for ( index1 = 0 ; index1 < N_POS_1 ; index1++ )  {
	for (index2 = 0 ; index2 < N_VEL_1 ; index2++ )  {
		for (index3 = 0 ; index3 < N_POS_2 ; index3++ )  {
			for (index4 = 0 ; index4 < N_VEL_2 ; index4++ )  {
	  			com_array[index1][index2][index3][index4]
					= UNOCCUPIED;
				counts[index1][index2][index3][index4] = 0;
	  		}
		}
	}
  }

}

/****************************************************************************/
/****************************************************************************/
STATE get_start()  {

  STATE temp_state;

  temp_state.pos_1 = START_POS_1;
  temp_state.vel_1 = START_VEL_1;
  temp_state.pos_2 = START_POS_2;
  temp_state.vel_2 = START_VEL_2;

  return( temp_state );

}

/****************************************************************************/
/****************************************************************************/
INDICES get_indices ( state )
	STATE state;
{

  float pos_encoder, float_pos_index;
  float vel_encoder, float_vel_index;
  INDICES indices;

  pos_encoder = state.pos_1 * COUNTS_PER_REV / 2 / PI;
  float_pos_index = ( pos_encoder - MIN_POS_ENCODER_1 ) * N_POS_1 /
        (MAX_POS_ENCODER_1 - MIN_POS_ENCODER_1);
  /* finds the closest integer */
  indices.pos_1 = ( float_pos_index + .5 ) / 1;

  vel_encoder = state.vel_1 * COUNTS_PER_REV / 2 / PI;
  float_vel_index = ( vel_encoder - MIN_VEL_ENCODER_1 ) * N_VEL_1 /
        (MAX_VEL_ENCODER_1 - MIN_VEL_ENCODER_1);
  /* finds the closest integer */
  indices.vel_1 = ( float_vel_index + .5 ) / 1;              

  pos_encoder = state.pos_2 * COUNTS_PER_REV / 2 / PI;
  float_pos_index = ( pos_encoder - MIN_POS_ENCODER_2 ) * N_POS_2 /
        (MAX_POS_ENCODER_2 - MIN_POS_ENCODER_2);
  /* finds the closest integer */
  indices.pos_2 = ( float_pos_index + .5 ) / 1;

  vel_encoder = state.vel_2 * COUNTS_PER_REV / 2 / PI;
  float_vel_index = ( vel_encoder - MIN_VEL_ENCODER_2 ) * N_VEL_2 /
        (MAX_VEL_ENCODER_2 - MIN_VEL_ENCODER_2);
  /* finds the closest integer */
  indices.vel_2 = ( float_vel_index + .5 ) / 1;              

  return( indices );

}

/****************************************************************************/
/****************************************************************************/
int pos_in_table( indices )
        INDICES indices;
{
/* checks to see if the current pos is in the table */

  if ( indices.pos_1 > -1 && indices.pos_1 < N_POS_1
        && indices.pos_2 > -1 && indices.pos_2 < N_POS_2 ) {
        return( TRUE );
  }
  else  {
        return( FALSE );
  }
  
}
  
/****************************************************************************/
/****************************************************************************/
INDICES get_in_table( indices )
        INDICES indices;
{
/* rolls position over to the other side of the table - assumes table is
symmetric */
  
  
  while( !pos_in_table( indices ) )  {
  
        if ( indices.pos_1 < N_POS_1 )
                indices.pos_1 = indices.pos_1 + N_POS_1;
        if ( indices.pos_1 >= N_POS_1 )
                indices.pos_1 = indices.pos_1 - N_POS_1;
        
        if ( indices.pos_2 < N_POS_2 )
                indices.pos_2 = indices.pos_2 + N_POS_2;
        if ( indices.pos_2 >= N_POS_2 )
                indices.pos_2 = indices.pos_2 - N_POS_2;
   
  }
  
  return( indices );

}
        
/****************************************************************************/
/****************************************************************************/
int in_table( indices )
	INDICES indices;
{

  if ( indices.pos_1 > -1 && indices.pos_1 < N_POS_1
	&& indices.vel_1 > -1 && indices.vel_1 < N_VEL_1
	&& indices.pos_2 > -1 && indices.pos_2 < N_POS_2
        && indices.vel_2 > -1 && indices.vel_2 < N_VEL_2 ) {
	return( TRUE );
  }
  else  {
	return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
STATE get_goal()  {

  STATE temp_state;

  temp_state.pos_1 = GOAL_POS_1;
  temp_state.vel_1 = GOAL_VEL_1;
  temp_state.pos_2 = GOAL_POS_2;
  temp_state.vel_2 = GOAL_VEL_2;

  return( temp_state );

}  

/****************************************************************************/
/****************************************************************************/
STATE get_prev_state( current_state, command )
	STATE current_state;
	int command;
{

/* equations are from Sutton 98 */

  /* parameters */
  float m1 = .232; /* mass link 1 (kg) */
  float m2 = .132; /* mass link 2 (kg) */
  float l1 = .2397; /* length link 1 (m) */
  float l2 = .2397; /* length link 2 (m) */
  float lc1 = .1199; /* axis to center of mass link 1 (m) */
  float lc2 = .0719; /* axis to center of mass link 2 (m) */
  float i1 = .006867; /* moment of inertia link 1 (kgm^2) */
  float i2 = .001162; /* moment of inertia link 2 (kgm^2) */
  float g = 9.81; /* gravity (m/s^2) */
  float motor_torque = .01785; /* torque of the motor */

  /* variables */
  float torque_sign = 0;
  float accel_1 = 0;
  float accel_2 = 0;
  float d1 = 0;
  float d2 = 0;
  float phi1 = 0;
  float phi2 = 0;
  float pos_1 = 0;
  float pos_2 = 0;
  float vel_1 = 0;
  float vel_2 = 0;
  STATE previous_state; 

  /* conversion for command */
  /* 0 = zero torque sent */
  /* 1 = negative torque sent */
  /* 2 = positive torque sent */
  switch( command )  {
        case 0:
                torque_sign = 0.00;
        break;
        case 1:
                torque_sign = -1.00;
        break;
        case 2:
                torque_sign = 1.00;
        break;                                           
  }

  /* change variables to make it easier */
  pos_1 = current_state.pos_1;
  pos_2 = current_state.pos_2;
  vel_1 = current_state.vel_1;
  vel_2 = current_state.vel_2;

  /* Calculate the state - notice it is previous since DELTA_TIME is < 0 */
  phi2 = m2 * lc2 * g * cos( pos_1 + pos_2 
	- PI / 2 );
  phi1 = - m2 * l1 * lc2 * pow( vel_2, 2 ) * sin( pos_2 )
	- 2 * m2 * l1 * lc2 * vel_2 * vel_1 * sin( pos_2 )
	+ ( m1 * lc1 + m2 * l1 ) * g * cos( pos_1 - PI/2 ) + phi2;
  d2 = m2 * ( pow( lc2, 2 ) + l1 * l2 * cos( pos_2 ) ) + i2;
  d1 = m1 * pow( lc1, 2 )
	+ m2 * ( pow( l1, 2 ) + pow( lc2, 2 ) + 2 * l1 * lc2 * cos( pos_2 ) )
	+ i1 + i2;
  accel_2 = 1 / ( m2 * pow( lc2, 2 ) + i2 - pow( d2, 2 ) / d1 ) 
	* ( torque_sign * motor_torque + d2 / d1 * phi1 - phi2 );
  accel_1 = - 1 / d1 * ( d2 * accel_2 + phi1 );

  /* get new state */
  previous_state.vel_1 = current_state.vel_1 + DELTA_TIME * accel_1;
  previous_state.pos_1 = current_state.pos_1 + DELTA_TIME 
	* current_state.vel_1
        + .5 * accel_1 * DELTA_TIME * DELTA_TIME;

  previous_state.vel_2 = current_state.vel_2 + DELTA_TIME * accel_2;
  previous_state.pos_2 = current_state.pos_2 + DELTA_TIME 
	* current_state.vel_2
        + .5 * accel_2 * DELTA_TIME * DELTA_TIME;

  return( previous_state );

}                                                         

/****************************************************************************/
/****************************************************************************/
int same_indices( indices1, indices2 )
	INDICES indices1;
	INDICES indices2;
{

  if ( ( indices1.pos_1 == indices2.pos_1 ) 
	&& ( indices1.vel_1 == indices2.vel_1 )
	&& ( indices1.pos_2 == indices2.pos_2 ) 
	&& ( indices1.vel_2 == indices2.vel_2 ) )  {
	return( TRUE );
  }
  else  {
	return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
int cell_unoccupied( indices )
	INDICES indices;
{

  if ( com_array[indices.pos_1][indices.vel_1][indices.pos_2][indices.vel_2]
	== UNOCCUPIED )  {
	return( TRUE );
  }
  else  {
	return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
mark_table( indices, command )
	INDICES indices;
	int command;
{

  com_array[indices.pos_1][indices.vel_1][indices.pos_2][indices.vel_2] 
	= command;

}

/****************************************************************************/
/****************************************************************************/
int cell_occupied( indices )
	INDICES indices;
{

  if ( com_array[indices.pos_1][indices.vel_1][indices.pos_2][indices.vel_2] 
	!= UNOCCUPIED )  {
	return( TRUE );
  }
  else  {
	return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
write_array_to_file() {

  FILE *fd;
  int index1, index2, index3, index4;

  if ( ( fd = fopen( "command.dat", "w" ) ) == NULL )  {
      fprintf( stderr, "cannot fopen file command.dat for write.\n" );
  }

  for ( index1 = 0 ; index1 < N_POS_1 ; index1++ )  {
        for ( index2 = 0 ; index2 < N_VEL_1 ; index2++ )  {
		for ( index3 = 0 ; index3 < N_POS_2 ; index3++ )  {
        		for ( index4 = 0 ; index4 < N_VEL_2 ; index4++ )  {
                		fprintf( fd, "%d ", 
				  com_array[index1][index2][index3][index4] );
			}
		        fprintf( fd, "\n" );
		}
	        fprintf( fd, "\n" );
        }
        fprintf( fd, "\n" );
  }

  fclose( fd );

}                 

/****************************************************************************/
/****************************************************************************/
write_info_file( depth )
	int depth;
{

/* writes info from the data run in the same order as the constant 
declaration of this program, except that the depth of the execution is 
listed first */

  FILE *fd;

  if ( ( fd = fopen( "command.info", "w" ) ) == NULL )  {
      fprintf( stderr, "cannot fopen file command.info for write.\n" );
  }

  fprintf( fd, "%d ", depth );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", N_POS_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", N_VEL_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", N_POS_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", N_VEL_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", UNOCCUPIED );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", START_POS_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", START_VEL_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", GOAL_POS_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", GOAL_VEL_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", START_POS_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", START_VEL_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", GOAL_POS_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", GOAL_VEL_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", COUNTS_PER_REV );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", PI );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MIN_POS_ENCODER_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MAX_POS_ENCODER_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MIN_VEL_ENCODER_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MAX_VEL_ENCODER_1 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MIN_POS_ENCODER_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MAX_POS_ENCODER_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MIN_VEL_ENCODER_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%d ", MAX_VEL_ENCODER_2 );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", DELTA_TIME );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", A );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", B );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", C );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", D );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", E );
  fprintf( fd, "\n" );
  fprintf( fd, "%f ", F );
  fprintf( fd, "\n" );

  fclose( fd );

}                 

/****************************************************************************/
/****************************************************************************/
print_queue( queue )
	QUEUE_NODE *queue;
{

  if ( queue != NULL )  {
	printf( "pos_1: %f vel_1: %f\n", queue->data.pos_1, 
		queue->data.vel_1 );
	printf( "pos_2: %f vel_2: %f\n", queue->data.pos_2, 
		queue->data.vel_2 );
 	print_queue( queue->next );
  }

}

/****************************************************************************/
/****************************************************************************/
print_state( state )
	STATE state;
{

	printf( "pos_1: %f vel_1: %f\n", state.pos_1, state.vel_1 );
	printf( "pos_2: %f vel_2: %f\n", state.pos_2, state.vel_2 );

}

/****************************************************************************/
/****************************************************************************/
print_indices( indices )
	INDICES indices;
{

	printf( "pos_1: %d vel_1: %d\n", indices.pos_1, indices.vel_1 );
	printf( "pos_2: %d vel_2: %d\n", indices.pos_2, indices.vel_2 );

}

/****************************************************************************/
/****************************************************************************/
print_table_command( indices )
	INDICES indices;
{

	printf( "table command: %d\n", 
	com_array[indices.pos_1][indices.vel_1][indices.pos_2][indices.vel_2]
	);

}

/****************************************************************************/
/****************************************************************************/
init_free_queue()  {

  free_queue = NULL;
 
  /* allocate memory for free_queue */
  free_queue = ( QUEUE_NODE * ) malloc( sizeof( QUEUE_NODE ) );
  if ( free_queue == NULL )  {
	printf ( "Memory allocation error" );                
	exit(-1);
  }                      

  free_queue->next = NULL;

}

/****************************************************************************/
/****************************************************************************/
QUEUE_NODE * add_level_marker( queue )
	QUEUE_NODE *queue;
{
  STATE fake_state;
  INFO fake_info;

  fake_state.pos_1 = 0;
  fake_state.vel_1 = 0;
  fake_state.pos_2 = 0;
  fake_state.vel_2 = 0;
  fake_info.command = 0;
  fake_info.same = LEVEL_MARKER;

  queue = add_to_end( queue, fake_state, fake_info );

  return( queue );

}

/****************************************************************************/
/****************************************************************************/
QUEUE_NODE * create_next_nodes( queue, current_state )
	QUEUE_NODE *queue;
	STATE current_state;
{

  int command;
  INFO info;
  STATE prev_state;
  INDICES prev_indices, current_indices;

  for (command = 0 ; command < 3 ; command++ )  {
	info.command = command;
	prev_state = get_prev_state( current_state, command );
	prev_indices = get_indices( prev_state );
	current_indices = get_indices( current_state );
	if ( in_table( prev_indices ) )  {
		if ( same_indices( current_indices, prev_indices ) )  {
			info.same = TRUE;
			queue = add_to_end( queue, prev_state, info );
		}
		else  {
			info.same = FALSE;
			queue = add_to_end( queue, prev_state, info );
		}
	}
  }

  return( queue );

}

/****************************************************************************/
/****************************************************************************/
STATE get_radians( indices )
	INDICES indices;
{

/* converts from indices to radians */

  float pos_encoder, float_pos_index;
  float vel_encoder, float_vel_index;
  STATE state;

  float_pos_index = indices.pos_1;
  pos_encoder = float_pos_index * (MAX_POS_ENCODER_1 - MIN_POS_ENCODER_1)
	/ N_POS_1;
  pos_encoder = pos_encoder + MIN_POS_ENCODER_1;
  state.pos_1 = pos_encoder * PI * 2 / COUNTS_PER_REV;

  float_vel_index = indices.vel_1;
  vel_encoder = float_vel_index * (MAX_VEL_ENCODER_1 - MIN_VEL_ENCODER_1)
	/ N_VEL_1;
  vel_encoder = vel_encoder + MIN_VEL_ENCODER_1;
  state.vel_1 = vel_encoder * PI * 2 / COUNTS_PER_REV;

  float_pos_index = indices.pos_2;
  pos_encoder = float_pos_index * (MAX_POS_ENCODER_2 - MIN_POS_ENCODER_2)
	/ N_POS_2;
  pos_encoder = pos_encoder + MIN_POS_ENCODER_2;
  state.pos_2 = pos_encoder * PI * 2 / COUNTS_PER_REV;

  float_vel_index = indices.vel_2;
  vel_encoder = float_vel_index * (MAX_VEL_ENCODER_2 - MIN_VEL_ENCODER_2)
	/ N_VEL_2;
  vel_encoder = vel_encoder + MIN_VEL_ENCODER_2;
  state.vel_2 = vel_encoder * PI * 2 / COUNTS_PER_REV;

  return( state );

}  

/****************************************************************************/
/****************************************************************************/
int num_queue_nodes( queue )
        QUEUE_NODE *queue;
/* counts the number fo nodes in the queue */
{

  if ( queue == NULL )
        return( 0 );
  else
        return( num_queue_nodes( queue->next ) + 1 );

}

/****************************************************************************/
/****************************************************************************/
int end_of_level_reached( queue )
	QUEUE_NODE *queue;
/* if it is looking at a level marker, the it returns TRUE */
{

  if ( queue == NULL )  {
	printf( "The queue is empty!" );
	exit( -1 );
  }
  else if ( queue->info.same == LEVEL_MARKER )  {
	return( TRUE );
  }
  else  {
	return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
int is_current_closer( current_state, node_state )
	STATE current_state;
	STATE node_state;
{

  STATE middle;
  INDICES indices;
  float current_pos_1_difference, current_vel_1_difference;
  float current_difference_1;
  float current_pos_2_difference, current_vel_2_difference;
  float current_difference_2;
  float current_difference;
  float node_pos_1_difference, node_vel_1_difference, node_difference_1;
  float node_pos_2_difference, node_vel_2_difference, node_difference_2;
  float node_difference;

  /* get the radians for the center of the cell */
  indices = get_indices( current_state );
  middle = get_radians( indices );

  /* get squared current difference for 1 */
  current_pos_1_difference = middle.pos_1 - current_state.pos_1;
  current_pos_1_difference = pow( current_pos_1_difference, 2 );
  current_vel_1_difference = middle.vel_1 - current_state.vel_1;
  current_vel_1_difference = pow( current_vel_1_difference, 2 );
  current_difference_1 = current_pos_1_difference + current_vel_1_difference;

  /* get squared current difference for 2 */
  current_pos_2_difference = middle.pos_2 - current_state.pos_2;
  current_pos_2_difference = pow( current_pos_2_difference, 2 );
  current_vel_2_difference = middle.vel_2 - current_state.vel_2;
  current_vel_2_difference = pow( current_vel_2_difference, 2 );
  current_difference_2 = current_pos_2_difference + current_vel_2_difference;

  current_difference = current_difference_1 + current_difference_2;

  /* get squared node difference for 1 */
  node_pos_1_difference = middle.pos_1 - node_state.pos_1;
  node_pos_1_difference = pow( node_pos_1_difference, 2 );
  node_vel_1_difference = middle.vel_1 - node_state.vel_1;
  node_vel_1_difference = pow( node_vel_1_difference, 2 );
  node_difference_1 = node_pos_1_difference + node_vel_1_difference;

  /* get squared node difference for 2 */
  node_pos_2_difference = middle.pos_2 - node_state.pos_2;
  node_pos_2_difference = pow( node_pos_2_difference, 2 );
  node_vel_2_difference = middle.vel_2 - node_state.vel_2;
  node_vel_2_difference = pow( node_vel_2_difference, 2 );
  node_difference_2 = node_pos_2_difference + node_vel_2_difference;

  node_difference = node_difference_1 + node_difference_2;

  if ( current_difference < node_difference )  {
	return( TRUE );
  }
  else  {
	return( FALSE );
  }

}

/****************************************************************************/
/****************************************************************************/
int closest( queue, current_state )
	QUEUE_NODE *queue;
	STATE current_state;
/* This function goes until the end of the level is reached, looks at 
the state there, determines if it is in the same cell in the lookup table 
as the current state, if it is, it determines whether it is closest to 
the center of the cell.  If any of the states that are in the same cell 
are closer than the current state is to the center of the cell, then a 
FALSE is returned.  Else a TRUE is returned, since the current state is 
closest to the middle of the cell. */
{

  int answer;
  INDICES current_indices, node_indices;

  if ( end_of_level_reached == FALSE )  {
	answer = closest( queue->next, current_state );
	current_indices = get_indices( current_state );
	node_indices = get_indices( queue->data );
	if ( same_indices( current_indices, node_indices ) == FALSE )  {
		/* ignore since it is not in the same cell */
		return( answer );
	}
	if ( same_indices( current_indices, node_indices ) )  {
		/* need to check which one is closest since in same cell */
		answer = is_current_closer( current_state, queue->data );
		return( answer );
	}
	if ( current_state.pos_1 == queue->data.pos_1 
		&& current_state.vel_1 == queue->data.vel_1
		&& current_state.pos_2 == queue->data.pos_2 
		&& current_state.vel_2 == queue->data.vel_2 )  {
		/* ignore since it is the original cell */
		return( answer );
	}
  }
  else {
	return( TRUE );  /* initializes as TRUE since at the level's end */
  }

}

/****************************************************************************/
/****************************************************************************/

main( argc, argv )
int argc;
char *argv[];
{
  QUEUE_NODE *queue;
  STATE start_state;
  STATE goal_state;
  STATE current_state;
  STATE prev_state;
  INDICES start_indices;
  INDICES goal_indices;
  INDICES current_indices;
  INDICES prev_indices;
  INDICES mark_indices;
  INDICES first_prev_indices, second_prev_indices;
  int depth;
  int end_not_reached;
  int not_too_deep;
  int max_depth;
  int command;
  int user_input;
  int level;
  INFO info;

  /* init stuff */
  init_array();
  depth = 0;
  end_not_reached = TRUE;
  not_too_deep = TRUE;
  queue = NULL;
  init_free_queue();
  level = 1;

  /* mark pos axis */
  /* if UNOCCUPIED, then it's not being used... */
  mark_indices.pos_1 = N_POS_1 - 10;
  mark_indices.vel_1 = 1;
  mark_indices.pos_2 = N_POS_2 - 10;
  mark_indices.vel_2 = 1;
  mark_table( mark_indices, UNOCCUPIED );

  /* get max queue depth = number of nodes visited and expanded */
  printf( "Enter the max depth.\n" );
  scanf( "%d", &max_depth );

  /* define start and get indices */
  /* notice it is not used in sim8 or later */
  start_state = get_start();
  start_indices = get_indices( start_state );
  if ( in_table( start_indices ) )  {
	/* don't do anything */
  }
  else  {
	printf( "Start not in table!!!\n" );
	exit( -1 );
  }

  /* define goal and assign in queue */
  goal_state = get_goal();
  info.same = FALSE;
  info.command = 0;
  queue = add_to_top( queue, goal_state, info );
  /* get indices and assign in table */
  goal_indices = get_indices( goal_state );
  if ( in_table( goal_indices ) )  {
	/* must leave unmarked since marking takes place post creation */
	/* mark_table( goal_indices, 0 ); */
  }
  else  {
	printf( "Goal not in table!!!\n" );
  }

  /* add a level marker */
  queue = add_level_marker( queue );

  /* do the cool stuff - ie find the path */
  while ( not_too_deep && queue != NULL )  {
	if ( end_of_level_reached( queue ) )  {
		printf( "End of level %d reached\n", level );		
		queue = add_level_marker( queue );
		level++;
	}
	else  {
		current_state = get_top_data( queue );
		info = get_top_info( queue );
		if ( info.same )  {
			current_indices = get_indices( current_state );
			current_indices = get_in_table( current_indices );
			/* add to the number of times it has been there */
			counts[current_indices.pos_1][current_indices.vel_1][current_indices.pos_2][current_indices.vel_2]++;
			if ( counts[current_indices.pos_1][current_indices.vel_1][current_indices.pos_2][current_indices.vel_2]
			  < MAX_STATE_DWELL )  {
			  if ( num_queue_nodes( queue ) > MAX_QUEUE_SIZE )  {
				/* just stick it on the front, don't 
				expand - a bad hack */
				printf( "Hit max node limit.\n" );
				queue = add_to_end( queue, current_state, info );
			  }
			  else  {	
				/* not too many nodes - expand */
				queue = 
				  create_next_nodes( queue, current_state );
			  }
			}
		}
		else  {
			current_indices = get_indices( current_state );
			current_indices = get_in_table( current_indices );
			if ( cell_unoccupied( current_indices ) )  {
                          if ( num_queue_nodes( queue ) > MAX_QUEUE_SIZE )  {
                                /* just stick it on the front, don't
                                expand - a bad hack */
				printf( "Hit max node limit.\n" );
                                queue = add_to_end( queue, current_state, info );
                          }
                          else  {
                                /* not too many nodes - expand */
				if ( closest( queue, current_state ) )  {
					mark_table( current_indices, 
						info.command );
					queue = create_next_nodes( queue,
						current_state );
				}
				else  {
					queue = create_next_nodes( queue,
						current_state );
				}					
			  }
			}
		}
	}	

	queue = pop_off_top( queue );
	depth++;
	
/*	if ( depth % INFO_INTERVAL == 0 )  {
		print_helpful_info();
	}
*/
	if ( depth % FILE_SAVE_INTERVAL == 0 )  {
		printf( "Writing output files at depth %d\n", depth );
		write_array_to_file();
		write_info_file( depth );
		/* print_helpful_info(); */
	}
	
	if ( cell_occupied( start_indices ) ) {
		end_not_reached = FALSE;
	}
	
	if ( depth >= max_depth )  {
		not_too_deep = FALSE;
	}
  }

  if ( queue == NULL )  {
	printf( "queue is empty at depth %d!!!\n", depth );
  }

  if ( end_not_reached == FALSE )  {
	printf( "Reached the start position.\n" );
  }

  printf( "Writing array to file.\n" );
  write_array_to_file();
  write_info_file( depth );

  print_helpful_info();

  return( 0 );
}

/*************************************************************************/
/****************************************************************************/
/*************************************************************************/
/****************************************************************************/
/****************************************************************************/
/***********************************************************************/
/**************************************************************************/
/****************************************************************************/
/*************************************************************************/
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/*************************************************************************/
/*************************************************************************/

print_helpful_info()
{
  int index1, index2, index3, index4;

  for ( index1 = 0 ; index1 < N_POS_1 ; index1++ )  {
	for (index2 = 0 ; index2 < N_VEL_1 ; index2++ )  {
		for ( index3 = 0 ; index3 < N_POS_2 ; index3++ )  {
			for (index4 = 0 ; index4 < N_VEL_2 ; index4++ )  {
	    			if ( counts[index1][index2][index3][index4]
				  > 10 )
	    			  printf("index %d %d %d %d: com %d: cnt %d\n",
		    		    index1, index2, index3, index4, 
				    com_array[index1][index2][index3][index4],
		    		    counts[index1][index2][index3][index4] );
	  		}
		}
    	}
  }

}

/****************************************************************************/




