////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Geometry programs using Cg
// sgreen 5/2006
//
// Email: sdkfeedback@nvidia.com
//
// Copyright (c) NVIDIA Corporation. All rights reserved.
////////////////////////////////////////////////////////////////////////////////////////////////////


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <map>

#include <GL/glew.h>
#include <GL/glut.h>

#include <nvGlutManipulators.h>
#include "trimesh.h"

#include <Cg/cg.h>
#include <Cg/cgGL.h>
#include <windows.h>  /* for QueryPerformanceCounter */
////////////////////////////////////////////////////////////////////////////////
//
//  globals
//
////////////////////////////////////////////////////////////////////////////////

#define DEMO_PATH "../gssc/shaders/"
using std::map;\

enum UIOption {
    OPTION_ANIMATE,
	OPTION_HIDDEN,
    OPTION_COUNT,
};
bool options[OPTION_COUNT];
map<char,UIOption> optionKeyMap;
nv::GlutExamine manipulator;

enum ModelMode {
	MODEL,
	MODEL_ADJ,
};

enum BUFFER_ID{
	VERTEX_BUFFER,
	NORMAL_BUFFER,
	PDIR1_BUFFER,
	PDIR2_BUFFER,
	PCURV_BUFFER,
	DCURV_BUFFER,
	ADJ_INDEX_BUFFER,
	TRI_INDEX_BUFFER,
	BUFFER_COUNT,
};

//
// model data
//
//nv::Model *model = 0;
TriMesh *model = 0;
char *model_filename = "horse.ply"; // default model
nv::vec3f modelMin, modelMax;
GLuint bufferID[BUFFER_COUNT];

// window size
int width = 800, height = 700;

float threshold = 0.8;

CGcontext cg_context; 
CGprofile cg_vprofile, cg_gprofile, cg_fprofile;

struct Program {
    const char *name;
    const char *filename;
    const char *vprog_entry, *gprog_entry, *fprog_entry;    // entry points for vertex, geometry and fragment programs
	ModelMode modelmode;
    CGprogram vprog, gprog, fprog;
};

Program prog[] = {
    { "contour + suggestive contour", DEMO_PATH "contour_sc.cg", "contour_vp", "contour_gp", 0, MODEL, 0, 0, 0 },

};

#define NPROGS sizeof(prog) / sizeof(Program)

#define CHECK_ERRORS  \
{ \
    GLenum err = glGetError(); \
    if (err) \
        printf( "Error 0x%x at line %d\n", err, __LINE__); \
}

Program *current_prog = &prog[0];

//////////////////////////////////////////////////////////////////////
//
// Callback function used for reporting Cg compile errors
//
//////////////////////////////////////////////////////////////////////
void cgErrorCallback()
{
    CGerror lastError = cgGetError();
    if(lastError)
    {
        printf("%s\n", cgGetErrorString(lastError));
        printf("%s\n", cgGetLastListing(cg_context));
        exit(1);
    }
}

void build_bufferobject()
{
	// Generate a buffer object
    glGenBuffers(BUFFER_COUNT, bufferID);
	
	GLfloat *arrValue;

	// create vertex buffer
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[VERTEX_BUFFER]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * model->vertices.size() * 3, NULL, GL_STATIC_DRAW);
	arrValue = (GLfloat *)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->vertices.size(); i++ )
	{
		*arrValue++ = model->vertices[i][0];
		*arrValue++ = model->vertices[i][1];
		*arrValue++ = model->vertices[i][2];
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);

	
	// create normal buffer
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[NORMAL_BUFFER]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * model->normals.size() * 3, NULL, GL_STATIC_DRAW);
	arrValue = (GLfloat *)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
	
	for ( unsigned int i = 0; i < model->normals.size(); i++ )
	{
		*arrValue++ = model->normals[i][0];
		*arrValue++ = model->normals[i][1];
		*arrValue++ = model->normals[i][2];
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);

	
	// create primary direction1 buffer
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[PDIR1_BUFFER]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * model->pdir1.size() * 3, NULL, GL_STATIC_DRAW);
	arrValue = (GLfloat *)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->pdir1.size(); i++ )
	{
		*arrValue++ = model->pdir1[i][0];
		*arrValue++ = model->pdir1[i][1];
		*arrValue++ = model->pdir1[i][2];
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);


	// create primary direction2 buffer
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[PDIR2_BUFFER]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * model->pdir2.size() * 3, NULL, GL_STATIC_DRAW);
	arrValue = (GLfloat *)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->pdir2.size(); i++ )
	{
		*arrValue++ = model->pdir2[i][0];
		*arrValue++ = model->pdir2[i][1];
		*arrValue++ = model->pdir2[i][2];
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);

	 
    // create primary curvatures buffer
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[PCURV_BUFFER]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * model->pcurv.size() * 2, NULL, GL_STATIC_DRAW);
	arrValue = (GLfloat *)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->pcurv.size(); i++ )
	{
		*arrValue++ = model->pcurv[i][0];
		*arrValue++ = model->pcurv[i][1];
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);


	// create derivatives of curvature buffer
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[DCURV_BUFFER]);
	glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * model->pcurv.size() * 4, NULL, GL_STATIC_DRAW);
	arrValue = (GLfloat *)glMapBuffer(GL_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->dcurv.size(); i++ )
	{
		*arrValue++ = model->dcurv[i][0];
		*arrValue++ = model->dcurv[i][1];
		*arrValue++ = model->dcurv[i][2];
		*arrValue++ = model->dcurv[i][3];
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);

	// create indices of triangles adj
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferID[ADJ_INDEX_BUFFER]);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLuint) * model->triangleAdjacency.size(), NULL, GL_STATIC_DRAW);
	GLuint *index = (GLuint *)glMapBuffer(GL_ELEMENT_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->triangleAdjacency.size(); i++ )
	{
		*index++ = model->triangleAdjacency[i];
	}
	glUnmapBuffer(GL_ELEMENT_ARRAY_BUFFER);

	// create indices of triangles adj
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferID[TRI_INDEX_BUFFER]);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLuint) * model->faces.size() * 3, NULL, GL_STATIC_DRAW);
	index = (GLuint *)glMapBuffer(GL_ELEMENT_ARRAY_BUFFER, GL_WRITE_ONLY);
	for ( unsigned int i = 0; i < model->faces.size(); i++ )
	{
		*index++ = model->faces[i][0];
		*index++ = model->faces[i][1];
		*index++ = model->faces[i][2];
	}
	glUnmapBuffer(GL_ELEMENT_ARRAY_BUFFER);

}


//////////////////////////////////////////////////////////////////////
//
//  Function for setting up initial OpenGL state. Also compiles Cg
// programs.
//
//////////////////////////////////////////////////////////////////////
void init_opengl()
{
    glewInit();
    
    if (!glewIsSupported(
        "GL_VERSION_2_0 "
        "GL_ARB_vertex_program "
        "GL_ARB_fragment_program "
        "GL_NV_gpu_program4 " // includes GL_NV_geometry_program4
        "GL_NV_texture_rectangle "
        "GL_ARB_texture_float "
		))
    {
        printf("Unable to load necessary extension(s)\n");
        printf("This sample requires:\n  OpenGL 2.0\n  GL_ARB_vertex_program\n  GL_ARB_fragment_program\n"
               "GL_NV_gpu_program4\n  GL_NV_texture_rectangle\n  GL_ARB_texture_float\nExiting...\n");
        exit(-1);
    }

    // get geometry program limits
    GLint max_output_vertices, max_total_output_components;
    glGetProgramivARB(GL_GEOMETRY_PROGRAM_NV, GL_MAX_PROGRAM_OUTPUT_VERTICES_NV, &max_output_vertices);
    glGetProgramivARB(GL_GEOMETRY_PROGRAM_NV, GL_MAX_PROGRAM_TOTAL_OUTPUT_COMPONENTS_NV, &max_total_output_components);
    printf("max output vertices = %d\n", max_output_vertices);
    printf("max total output components = %d\n", max_total_output_components);

    // set normal rendering options
    glEnable(GL_DEPTH_TEST);
    glDepthFunc( GL_LEQUAL);
    glClearColor(1,1,1,0);

    //glDisable(GL_CULL_FACE);
	glDisable(GL_LIGHT0);

	// Create Cg Context
    cg_context = cgCreateContext();
    cgSetErrorCallback(cgErrorCallback);

    // load Cg programs
    cg_vprofile = cgGLGetLatestProfile(CG_GL_VERTEX);
    cg_gprofile = cgGLGetLatestProfile(CG_GL_GEOMETRY);
    cg_fprofile = cgGLGetLatestProfile(CG_GL_FRAGMENT);

    const char *args[] = {
        "-unroll",
        "all",
        0,
    };

    // load programs
    for(int i=0; i<NPROGS; i++) {
        Program &p = prog[i];

        p.vprog = cgCreateProgramFromFile(cg_context, CG_SOURCE, p.filename, cg_vprofile, p.vprog_entry, 0);
        cgGLLoadProgram(p.vprog);

        p.gprog = cgCreateProgramFromFile(cg_context, CG_SOURCE, p.filename, cg_gprofile, p.gprog_entry, args);
        cgGLLoadProgram(p.gprog);

        if (p.fprog_entry) {
            p.fprog = cgCreateProgramFromFile(cg_context, CG_SOURCE, p.filename, cg_fprofile, p.fprog_entry, 0);
            cgGLLoadProgram(p.fprog);
        }
    }

	glGenBuffers(2, bufferID);

    glutReportErrors();
}



//////////////////////////////////////////////////////////////////////
void DrawWithAdjacency() 
{
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferID[ADJ_INDEX_BUFFER]);
	glEnableClientState(GL_INDEX_ARRAY);

	glDrawElements( GL_TRIANGLES_ADJACENCY_EXT, (GLsizei)model->triangleAdjacency.size(), GL_UNSIGNED_INT, 0);

	glDisableClientState(GL_INDEX_ARRAY);
}

void DrawTriangle()
{
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferID[TRI_INDEX_BUFFER]);
	glEnableClientState(GL_INDEX_ARRAY);

	glDrawElements( GL_TRIANGLES, (GLsizei)model->faces.size()*3, GL_UNSIGNED_INT, 0);

	glDisableClientState(GL_INDEX_ARRAY);
}


void DrawOrg() 
{
    float factor = 1.0f / model->bsphere.r;  //2.0f/nv::length(modelMax - modelMin);

    glMatrixMode( GL_MODELVIEW);
    glPushMatrix();
    glScalef( factor, factor, factor);
    glTranslatef( -model->bsphere.center[0], -model->bsphere.center[1], -model->bsphere.center[2]);
    
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[VERTEX_BUFFER]);
    glVertexPointer(3, GL_FLOAT, 0, 0);
	
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[NORMAL_BUFFER]);
    glNormalPointer(GL_FLOAT, 0, 0);
    
    glEnableClientState( GL_VERTEX_ARRAY);
    glEnableClientState( GL_NORMAL_ARRAY);

    CHECK_ERRORS;
		
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, bufferID[TRI_INDEX_BUFFER]);
	glEnableClientState( GL_INDEX_ARRAY);
	glDrawElements( GL_TRIANGLES, (GLsizei)model->faces.size()*3, GL_UNSIGNED_INT, 0);

    CHECK_ERRORS;
	glDisableClientState(GL_INDEX_ARRAY);
    glDisableClientState( GL_VERTEX_ARRAY);
    glDisableClientState( GL_NORMAL_ARRAY);
	glPopMatrix();

}

void draw_model()
{
	switch(current_prog->modelmode) {
	case MODEL:
        if (model) {
            DrawTriangle();
        } 
        break;

    case MODEL_ADJ:
        if (model) {
            DrawWithAdjacency();
        } 
        break;
	}
	CHECK_ERRORS;
}

void draw_geometry()
{
   //nv::vec3f center = (modelMax + modelMin) * 0.5f;
	float factor = 1.0f / model->bsphere.r;  //2.0f/nv::length(modelMax - modelMin);

    glMatrixMode( GL_MODELVIEW);
    glPushMatrix();
    glScalef( factor, factor, factor);
    glTranslatef( -model->bsphere.center[0], -model->bsphere.center[1], -model->bsphere.center[2]);
    
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[VERTEX_BUFFER]);
    glVertexPointer(3, GL_FLOAT, 0, 0);
	//glVertexPointer( 3, GL_FLOAT, 0, &model->vertices[0][0]);
    
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[NORMAL_BUFFER]);
    glNormalPointer(GL_FLOAT, 0, 0);
    //glNormalPointer( GL_FLOAT, 0, &model->normals[0][0]);

    glEnableClientState( GL_VERTEX_ARRAY);
    glEnableClientState( GL_NORMAL_ARRAY);

    CHECK_ERRORS;
	
	glClientActiveTexture(GL_TEXTURE0);
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[PDIR1_BUFFER]);
	glTexCoordPointer( 3, GL_FLOAT,0, 0);
    //glTexCoordPointer( 3, GL_FLOAT,0, &model->pdir1[0][0]);
    glEnableClientState( GL_TEXTURE_COORD_ARRAY);
    
	glClientActiveTexture(GL_TEXTURE1);
    glBindBuffer(GL_ARRAY_BUFFER, bufferID[PDIR2_BUFFER]);
	glTexCoordPointer( 3, GL_FLOAT,0, 0);
	//glTexCoordPointer( 3, GL_FLOAT,0, &model->pdir2[0][0]);
    glEnableClientState( GL_TEXTURE_COORD_ARRAY);

	glClientActiveTexture(GL_TEXTURE2);
	glBindBuffer(GL_ARRAY_BUFFER, bufferID[PCURV_BUFFER]);
	glTexCoordPointer( 2, GL_FLOAT,0, 0);
    //glTexCoordPointer( 2, GL_FLOAT,0, &model->pcurv[0][0]);
    glEnableClientState( GL_TEXTURE_COORD_ARRAY);

	glClientActiveTexture(GL_TEXTURE3);
    glBindBuffer(GL_ARRAY_BUFFER, bufferID[DCURV_BUFFER]);
	glTexCoordPointer( 4, GL_FLOAT,0, 0);
	//glTexCoordPointer( 4, GL_FLOAT,0, &model->dcurv[0][0]);
    glEnableClientState( GL_TEXTURE_COORD_ARRAY);

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_LINE_SMOOTH);
	//glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glLineWidth(1.5);
	
	draw_model();

    CHECK_ERRORS;
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_BLEND);
	glDisableClientState( GL_VERTEX_ARRAY);
    glDisableClientState( GL_NORMAL_ARRAY);
	glClientActiveTexture(GL_TEXTURE0);
    glDisableClientState( GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture(GL_TEXTURE1);
    glDisableClientState( GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture(GL_TEXTURE2);
    glDisableClientState( GL_TEXTURE_COORD_ARRAY);
	glClientActiveTexture(GL_TEXTURE3);
    glDisableClientState( GL_TEXTURE_COORD_ARRAY);

    glPopMatrix();
}


void drawFPS(double fpsRate)
{
  GLubyte dummy;
  char buffer[200], *c;

  glPushMatrix();
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
      glLoadIdentity();
      glOrtho(0, 1, 1, 0, -1, 1);
      glDisable(GL_DEPTH_TEST);
      glColor3f(0,0,0);
      glRasterPos2f(1,1);
      glBitmap(0, 0, 0, 0, -10*9, 15, &dummy);
      sprintf(buffer, "fps %0.1f", fpsRate);
      for (c = buffer; *c != '\0'; c++)
        glutBitmapCharacter(GLUT_BITMAP_9_BY_15, *c);
      glEnable(GL_DEPTH_TEST);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

void drawThreshold()
{
  GLubyte dummy;
  char buffer[200], *c;

  glPushMatrix();
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
      glLoadIdentity();
      glOrtho(0, 1, 1, 0, -1, 1);
      glDisable(GL_DEPTH_TEST);
      glColor3f(0,0,0);
      glRasterPos2f(1,1);
      glBitmap(0, 0, 0, 0, -16*9, 35, &dummy);
      sprintf(buffer, "threshold %0.3f", threshold);
      for (c = buffer; *c != '\0'; c++)
        glutBitmapCharacter(GLUT_BITMAP_9_BY_15, *c);
      glEnable(GL_DEPTH_TEST);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}
static int myDrawFPS = 1;

static void handleFPS(void)
{
#ifdef _WIN32
  //static void drawFPS(double fpsRate);
  static __int64 freq = 0;
  static __int64 lastCount;  /* Timer count for last fps update */
  static int frameCount;     /* Number of frames for timing */
  static int fpsRate;        /* Current frames per second rate */
  static double lastFpsRate;
  __int64 newCount;
  
  if (!freq)
    QueryPerformanceFrequency((LARGE_INTEGER*) &freq);
  
  /* Update the frames per second count if we have gone past at least
     a second since the last update. */
  QueryPerformanceCounter((LARGE_INTEGER*) &newCount);
  frameCount++;
  if (((newCount - lastCount) > freq) && drawFPS) {
    double fpsRate;
    
    fpsRate = (double) (freq * (__int64) frameCount) /
              (double) (newCount - lastCount);
    lastCount = newCount;
    frameCount = 0;
    lastFpsRate = fpsRate;
  }
  if (myDrawFPS)
    drawFPS(lastFpsRate);
#endif
}


//////////////////////////////////////////////////////////////////////
void display()
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    manipulator.applyTransform();
	
	
    //glPolygonMode(GL_FRONT_AND_BACK, options[OPTION_DISPLAY_WIREFRAME] ? GL_LINE : GL_FILL);


	if ( options[OPTION_HIDDEN] )
	{
		// Draw the mesh
		//glDepthMask(GL_TRUE);
		glDepthFunc(GL_LESS);
		//glEnable(GL_DEPTH_TEST);
		glPolygonOffset(5.0f, 0.0f);
		glEnable(GL_POLYGON_OFFSET_FILL);
		glPolygonMode(GL_FRONT, GL_FILL);
		DrawOrg();
		glClear(GL_COLOR_BUFFER_BIT);
	}		
	
	// bind programs
    {
		cgGLBindProgram(current_prog->vprog);
		cgGLEnableProfile(cg_vprofile);

		cgGLBindProgram(current_prog->gprog);
        cgGLEnableProfile(cg_gprofile);

	    CGparameter param = cgGetNamedParameter(current_prog->gprog, "threshold");
		if (param) {
			cgGLSetParameter1f(param, threshold);
		}
	}

    CHECK_ERRORS;

	glDepthMask(GL_FALSE);

	//glActiveTexture(GL_TEXTURE0);
	//glBindTexture(GL_TEXTURE_3D, noise_tex);

    //glActiveTexture(GL_TEXTURE1);
    //glBindTexture(GL_TEXTURE_3D, noise_tex);

	//glActiveTexture(GL_TEXTURE2);
    //glBindTexture(GL_TEXTURE_RECTANGLE_NV, rand_tex);

	//glActiveTexture(GL_TEXTURE3);
    //glBindTexture(GL_TEXTURE_RECTANGLE_NV, rand_tex);

	//glActiveTexture(GL_TEXTURE4);
    //glBindTexture(GL_TEXTURE_RECTANGLE_NV, rand_tex);



    glColor3f(1.0, 1.0, 1.0);
	draw_geometry();
	
	glDepthMask(GL_TRUE);

    cgGLDisableProfile(cg_vprofile);
    cgGLDisableProfile(cg_gprofile);
    cgGLDisableProfile(cg_fprofile);

    CHECK_ERRORS;

	handleFPS();
	drawThreshold();
    glutReportErrors();
    glutSwapBuffers();
}


//////////////////////////////////////////////////////////////////////
void idle()
{
    if ( options[OPTION_ANIMATE]) {
        manipulator.idle();
    }

    glutPostRedisplay();
}


//////////////////////////////////////////////////////////////////////
void key(unsigned char k, int x, int y)
{
	k = tolower(k);

    if (optionKeyMap.find(k) != optionKeyMap.end())
        options[optionKeyMap[k]] = !options[optionKeyMap[k]];

    switch(k) {
        case 27:
        case 'q':
            exit(0);
            break;

        case '=':
        case '+':
            threshold += 0.001;
			threshold = threshold > 1.0f ? 1.0f : threshold;
            break;

        case '-':
		case '_':
            threshold -= 0.001;
			threshold = threshold < 0.0f ? 0.0f : threshold;
            break;
		default:
			break;
    }

    
	glutPostRedisplay();
}


//////////////////////////////////////////////////////////////////////
void resize(int w, int h)
{
    if (h == 0) h = 1;

    glViewport(0, 0, w, h);
    
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    
    gluPerspective(60.0, (GLfloat)w/(GLfloat)h, 0.1, 100.0);
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    manipulator.reshape(w, h);

    width = w;
    height = h;
}


//////////////////////////////////////////////////////////////////////
void mouse(int button, int state, int x, int y)
{
    manipulator.mouse(button, state, x, y);
}


//////////////////////////////////////////////////////////////////////
void motion(int x, int y)
{
    manipulator.motion(x, y);
}

//////////////////////////////////////////////////////////////////////
void main_menu(int i)
{
  key((unsigned char) i, 0, 0);
}

//////////////////////////////////////////////////////////////////////
void prog_menu(int i)
{
    current_prog = &prog[i];
}


//////////////////////////////////////////////////////////////////////
void init_menus()
{
    int prog_menu_id = glutCreateMenu(prog_menu);
    for(int i=0; i<NPROGS; i++) {
        glutAddMenuEntry( prog[i].name, i);
    }

    glutCreateMenu(main_menu);
    glutAddSubMenu("Program", prog_menu_id);
	glutAddMenuEntry("Quit (esc)", '\033');
	glutAddMenuEntry("Increase threshold", '+');
	glutAddMenuEntry("Decrease threshold", '-');
	glutAddMenuEntry("Toggle hidden lines", 'h');
    glutAttachMenu(GLUT_RIGHT_BUTTON);
}


//////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	glutInit(&argc, argv);
	glutInitWindowSize( width, height);
	glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGB);
   
	glutCreateWindow("Suggestive Contours");

	init_opengl();

    // configure a simple controller
    manipulator.setDollyActivate( GLUT_LEFT_BUTTON, GLUT_ACTIVE_CTRL);
    manipulator.setPanActivate( GLUT_LEFT_BUTTON, GLUT_ACTIVE_SHIFT);
    manipulator.setDollyPosition( -2.0f);

    // setup event callback functions
	glutDisplayFunc(display);
    glutMouseFunc(mouse);
    glutMotionFunc(motion);
    glutIdleFunc(idle);
    glutKeyboardFunc(key);
    glutReshapeFunc(resize);
    init_menus();

    // accept alternate models via the command line
    if (argc > 1) {
        model_filename = argv[1];
    }

	model = TriMesh::read(model_filename);
	if (!model)
	{
	    fprintf(stderr, "Error loading model '%s'\n", model_filename);
        delete model;
        model = 0;
		return -1;
    }
	
	//precomputing values of this model
	model->need_tstrips();
	model->need_bsphere();
	model->need_normals();
	model->need_curvatures();
	model->need_dcurv();
	//model->need_triangleAdjacency();
	//model->need_tstripsAdjacency();
 
	build_bufferobject();

    
  //configure the options
    optionKeyMap[' '] = OPTION_ANIMATE;
    options[OPTION_ANIMATE] = false;

	optionKeyMap['h'] = OPTION_HIDDEN;
    options[OPTION_HIDDEN] = true;


    //print the help info
    printf( "cg_geometry_program - sample showing the usage of geometry programs with Cg\n");
    printf( "  Commands:\n");
    printf( "    q / [ESC] - Quit the application\n");
    printf( "    [SPACE]   - Toggle continuous animation\n");
	printf( "    + / =     - Increase threshold\n");
	printf( "    _ / -     - Decrease threshold\n");
	printf( "      h       - Toggle hidden lines\n");
		
	glutMainLoop();

	return 0;
}