/*
Szymon Rusinkiewicz
Princeton University

TriMesh_connectivity.cc
Manipulate data structures that describe connectivity between faces and verts.
*/


#include <stdio.h>
#include "TriMesh.h"
#include <algorithm>
using std::find;


// Find the direct neighbors of each vertex
void TriMesh::need_neighbors()
{
	if (!neighbors.empty())
		return;
	need_faces();

	dprintf("Finding vertex neighbors... ");
	int nv = vertices.size(), nf = faces.size();

	vector<int> numneighbors(nv);
	for (int i = 0; i < nf; i++) {
		numneighbors[faces[i][0]]++;
		numneighbors[faces[i][1]]++;
		numneighbors[faces[i][2]]++;
	}

	neighbors.resize(nv);
	for (int i = 0; i < nv; i++)
		neighbors[i].reserve(numneighbors[i]+2); // Slop for boundaries

	for (int i = 0; i < nf; i++) {
		for (int j = 0; j < 3; j++) {
			vector<int> &me = neighbors[faces[i][j]];
			int n1 = faces[i][(j+1)%3];
			int n2 = faces[i][(j+2)%3];
			if (find(me.begin(), me.end(), n1) == me.end())
				me.push_back(n1);
			if (find(me.begin(), me.end(), n2) == me.end())
				me.push_back(n2);
		}
	}

	dprintf("Done.\n");
}


// Find the faces touching each vertex
void TriMesh::need_adjacentfaces()
{
	if (!adjacentfaces.empty())
		return;
	need_faces();

	dprintf("Finding vertex to triangle maps... ");
	int nv = vertices.size(), nf = faces.size();

	vector<int> numadjacentfaces(nv);
	for (int i = 0; i < nf; i++) {
		numadjacentfaces[faces[i][0]]++;
		numadjacentfaces[faces[i][1]]++;
		numadjacentfaces[faces[i][2]]++;
	}

	adjacentfaces.resize(vertices.size());
	for (int i = 0; i < nv; i++)
		adjacentfaces[i].reserve(numadjacentfaces[i]);

	for (int i = 0; i < nf; i++) {
		for (int j = 0; j < 3; j++)
			adjacentfaces[faces[i][j]].push_back(i);
	}

	dprintf("Done.\n");
}


// Find the face across each edge from each other face (-1 on boundary)
// If topology is bad, not necessarily what one would expect...
void TriMesh::need_across_edge()
{
	if (!across_edge.empty())
		return;
	need_adjacentfaces();

	dprintf("Finding across-edge maps... ");

	int nf = faces.size();
	across_edge.resize(nf, Face(-1,-1,-1));
	opposite_vertex.resize(nf, Face(-1,-1,-1));

	for (int i = 0; i < nf; i++) {
		for (int j = 0; j < 3; j++) {
			if (across_edge[i][j] != -1)
				continue;
			int v1 = faces[i][(j+1)%3];
			int v2 = faces[i][(j+2)%3];
			const vector<int> &a1 = adjacentfaces[v1];
			const vector<int> &a2 = adjacentfaces[v2];
			for (int k1 = 0; k1 < a1.size(); k1++) {
				int other = a1[k1];
				if (other == i)
					continue;
				vector<int>::const_iterator it =
					find(a2.begin(), a2.end(), other);
				if (it == a2.end())
					continue;
				int ind = (faces[other].indexof(v1)+1)%3;
				if (faces[other][(ind+1)%3] != v2)
					continue;
				across_edge[i][j] = other;
				across_edge[other][ind] = i;
				opposite_vertex[i][j] = faces[other][ind];
				opposite_vertex[other][ind] = faces[i][j];
				break;
			}
		}
	}

	dprintf("Done.\n");
}

// for geometry shader
void TriMesh::need_triangleAdjacency()
{
	if (!triangleAdjacency.empty())
		return;

	need_across_edge();

	dprintf("Making triangles with adjacency... ");

	int nf = faces.size();
	
	for ( int i = 0; i < nf; i++ )
	{
		for ( int j = 0; j < 3; j++ )
		{
			triangleAdjacency.push_back(faces[i][j]);
			int other_face = across_edge[i][(j+2)%3];
			int k = 0;
			for(; k < 3; k++ )
			{
				if ( across_edge[other_face][k] == i )
				{
					triangleAdjacency.push_back(faces[other_face][k]);
					break;
				}
			}
			if ( k >= 3)
			{
				dprintf("Error in making triangles with adjacencies.");
			}

		}
	}

	dprintf("Done.\n");
}

// for geometry shader
void TriMesh::need_tstripsAdjacency()
{	
	if ( !tstripsAdjacency.empty() )
	{
		return;
	}

	need_tstrips();
	need_faces();
	need_across_edge();


	dprintf("Making triangle strips with adjacency... ");

	int i = 0;
	vector<int> facesInStrips;

	while (i < tstrips.size()) 
	{
		int nfaces = tstrips[i] - 2;
		i += tstrips[i] + 1;
		facesInStrips.push_back(nfaces);
	}

	tstripsAdjacency.reserve(facesInStrips.size());

	int faceIndex = 0;
	for ( int i = 0; i < facesInStrips.size(); i++ )
	{
		int facenum = facesInStrips[i];
		vector<int> tadj;

		// part of the first triangle and adjacency
		// 1
		tadj.push_back(faces[faceIndex][0]);
		
		// 2
		tadj.push_back(opposite_vertex[faceIndex][2]);
		
		// 3
		tadj.push_back(faces[faceIndex][1]);
	
		for ( int j = 0; j < facenum; j++ )
		{
			// middle triangles
			if ( j % 2 == 0 )
			{
				tadj.push_back(opposite_vertex[faceIndex][1]);
			}
			else
			{
				tadj.push_back(opposite_vertex[faceIndex][0]);
			}
			tadj.push_back(faces[faceIndex][2]);

			faceIndex++;
		}

		// adjacency of the last triangle
		if ( facenum % 2 == 0 ) // even
		{
			tadj.push_back(opposite_vertex[faceIndex-1][1]);
		}
		else
		{
			tadj.push_back(opposite_vertex[faceIndex-1][0]);
		}

		tstripsAdjacency.push_back(tadj);
	}

	dprintf("Done.\n");
}