/*
 * This program does R-W-R. It reads a value from the mem, modifies it, writes it back, and reads it again
 * The value it reads is the value at address 0x1 in the input file (proj15.hex). This value is 0x6403
 * Read the README for details on the memory interface
 * */


#include <fstream>
#include <chdl/chdl.h>
#include <chdl/cassign.h>
#include "random_delay_mem.h"
#include <chdl/lfsr.h>

#define B 4
#define N 4
#define A 7
#define T 5
#define I 6
#define SZ 7

using namespace std;
using namespace chdl;

//This is a substitute run function which calls advance_eq, which in turn advances the internal memory state. 
void run_with_mem(ofstream& vcdout, unsigned long long cycles){
    print_vcd_header(vcdout);
    print_time(vcdout);

    for (unsigned i = 0; i < cycles; ++i) {
        print_taps(vcdout);
        for (auto &t : tickables()[0]) t->pre_tick(0);
        for (auto &t : tickables()[0]) t->tick(0);
        advance_eq(sim_time());
        for (auto &t : tickables()[0]) t->tock(0);
        for (auto &t : tickables()[0]) t->post_tock(0);
        now[0]++;
        print_time(vcdout);
    }
    call_final_funcs();
}

int main() {

    //Seed the random number
    srand(0);

    // Tie wires to req structure
    node req_valid,req_ready,req_wr;
    bvec<A> req_addr;
    bvec<N> req_mask;
    vec<N, bvec<B> > req_data;
    bvec<I> req_id;
    mem_req<B, N, A, I> req;
    _(req, "valid") = req_valid;
    _(req, "ready") = req_ready; // This is set internally. Just CHECK this. Don't SET it.
    _(_(req, "contents"), "wr") = req_wr;
    _(_(req, "contents"), "mask") = req_mask;
    _(_(req, "contents"), "addr") = req_addr;
    _(_(req, "contents"), "data") = req_data;
    _(_(req, "contents"), "id") = req_id;
    _(_(req, "contents"), "llsc") = Lit(0);


    // Tie wires to resp structure
    node resp_valid, resp_ready;
    vec<N, bvec<B> > resp_data;
    bvec<I> resp_id;
    mem_resp<B, N, I> resp;
    _(resp, "valid") = resp_valid;
    _(_(resp, "contents"), "data") = resp_data;
    _(_(resp, "contents"), "id") = resp_id;
    _(resp, "ready") = resp_ready; 
    resp_ready = !AndN(Lfsr<2, 31, 3, 1234>()); // Set resp_ready based on random number generator

    // Status flags
    node read_1_valid, write_1_valid, read_2_valid;
    node read_1_issued, write_1_issued, read_2_issued;
    node read_1_received, write_1_received, read_2_received, write_val_ready;

    // Read from address 0x1 
    req_addr = Lit<A>(1);   

    // Set unique ids for each request
    bvec<I> read_1_id, write_1_id, read_2_id;
    read_1_id = Lit<I>(1);
    write_1_id = Lit<I>(2);
    read_2_id = Lit<I>(3);

    // Read the entire line.
    req_mask = Lit<N>((1<<(N+1)) - 1);

    // Set up dependencies 
    read_1_valid = !read_1_issued;
    write_1_valid = !write_1_issued && write_val_ready;
    read_2_valid = write_1_received && !read_2_issued; 

    read_1_issued = Wreg(req_valid && read_1_valid, Lit(1));
    write_1_issued = Wreg(req_valid && write_1_valid, Lit(1));
    read_2_issued = Wreg(req_valid && read_2_valid, Lit(1));

    read_1_received = resp_ready && resp_valid && resp_id == read_1_id; 
    write_1_received = resp_ready && resp_valid && resp_id == write_1_id; 
    read_2_received = resp_ready && resp_valid && resp_id == read_2_id; 

    // Read data from first read req to local register. Value written to reg 1 cycle after read_1_received
    bvec<N*B> read_1_reg, write_val, read_2_reg;
    for (unsigned i =0 ; i < N ; i++){
        for (unsigned j =0 ; j < B ; j++){
            read_1_reg[i*N+j] = Wreg(read_1_received,resp_data[i][j]);
        }
    }

    // Set the write value to read_1_value + 1 
    write_val_ready = Wreg(Reg(read_1_received),Lit(1)); 
    write_val = Mux(write_val_ready, Lit<B*N>(0), read_1_reg + Lit<B*N>(1)); 

    // Read data from the second read req to local register. 
    for (unsigned i =0 ; i < N ; i++){
        for (unsigned j =0 ; j < B ; j++){
            read_2_reg[i*N+j] = Wreg(read_2_received,resp_data[i][j]);
        }
    }
    
    // Set req fields
    req_valid = Mux(req_ready, Lit(0), read_1_valid || write_1_valid || read_2_valid);
    Cassign(req_id).
        IF(read_1_valid,read_1_id).
        IF(write_1_valid,write_1_id).  
        IF(read_2_valid,read_2_id);

    Cassign(req_wr).
        IF(read_1_valid,Lit(0)).
        IF(write_1_valid,Lit(1)).  
        IF(read_2_valid,Lit(0));
    
    for (unsigned i =0 ; i < N ; i++){
        for (unsigned j =0 ; j < B ; j++){
            req_data[i][j] = Mux(write_1_valid, Lit(0), write_val[i*N+j]);
        }
    }

    RandomDelayMem<SZ, T, B, N, A, I>(resp,req,"proj15.hex");

    TAP(read_1_valid);TAP(read_2_valid); TAP(write_1_valid);
    TAP(read_1_issued); TAP(read_2_issued); TAP(write_1_issued);
    TAP(read_1_received);TAP(read_2_received); TAP(write_1_received);
    TAP(read_1_reg);TAP(write_val);TAP(write_val_ready);TAP(read_2_reg); 

    TAP(resp);
    TAP(req);
    optimize();
    ofstream vcdout("memtest.vcd");
    run_with_mem(vcdout, 100);

    return 0;
}
